// ==========================================
// PRESELL AVIATOR - Raspadinha Sequencial
// ==========================================

// ==========================================
// REDIRECIONAMENTO MOBILE - PRIMEIRA PRIORIDADE
// ==========================================

// Função para detectar dispositivo móvel REAL (não F12 ou simuladores)
function isRealMobileDevice() {
    const hasTouch = 'ontouchstart' in window || navigator.maxTouchPoints > 0;
    const mobileUA = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
    const narrowScreen = window.innerWidth < 768;
    const isNotDevTools = !navigator.userAgent.includes('Chrome') || 
                          (navigator.userAgent.includes('Mobile') && 
                           (navigator.platform === 'iPhone' || 
                            navigator.platform === 'Android' ||
                            navigator.platform === 'Linux armv81'));
    
    return hasTouch && mobileUA && narrowScreen && isNotDevTools;
}

// Função para extrair UTMs da URL atual
function getUTMParams() {
    const params = new URLSearchParams(window.location.search);
    const utms = {};
    
    for (const [key, value] of params.entries()) {
        if (key.startsWith('utm_') || key === 'click_id') {
            utms[key] = value;
        }
    }
    
    return utms;
}

// Função para construir URL com UTMs
function buildURLWithUTMs(baseURL) {
    const utms = getUTMParams();
    const url = new URL(baseURL, window.location.origin);
    
    Object.keys(utms).forEach(key => {
        url.searchParams.set(key, utms[key]);
    });
    
    return url.toString();
}

// EXECUTAR VERIFICAÇÃO IMEDIATAMENTE
(function() {
    console.log('🔍 Verificando dispositivo...');
    
    const isMobile = isRealMobileDevice();
    
    if (isMobile) {
        console.log('📱 Mobile detectado! Redirecionando...');
        const redirectURL = buildURLWithUTMs('/teste/');
        window.location.href = redirectURL;
        throw new Error('Redirecionamento mobile');
    }
    
    console.log('💻 Desktop detectado');
})();

// ==========================================
// RASPADINHA SEQUENCIAL
// ==========================================

// Configuração das raspadinhas
const SCRATCH_CARDS = [
    { prize: 0, type: 'loss' },     // 1ª: Não ganha nada
    { prize: 5, type: 'win' },      // 2ª: R$ 5,00
    { prize: 10, type: 'win' }      // 3ª: R$ 10,00
];

let currentCardIndex = 0;
let totalPrizeValue = 0;
const REVEAL_THRESHOLD = 40; // 40% raspado

// Inicializar quando DOM carregar
window.addEventListener('DOMContentLoaded', () => {
    initSequentialScratch();
});

function initSequentialScratch() {
    loadCard(currentCardIndex);
}

function loadCard(index) {
    const canvas = document.getElementById('scratchCanvas');
    const ctx = canvas.getContext('2d');
    const wrapper = canvas.parentElement;
    const scratchContent = document.getElementById('scratchContent');
    const progressCount = document.getElementById('progressCount');
    const progressFill = document.getElementById('progressFill');
    const instructionText = document.getElementById('instructionText');
    
    // Criar novo canvas para evitar conflitos de eventos
    const oldCanvas = canvas;
    const newCanvas = oldCanvas.cloneNode(true);
    oldCanvas.parentNode.replaceChild(newCanvas, oldCanvas);
    
    const newCtx = newCanvas.getContext('2d');
    
    // Resetar canvas
    scratchContent.style.display = 'none';
    newCanvas.style.opacity = '1';
    newCanvas.style.display = 'block';
    
    // Adicionar animação de entrada
    wrapper.classList.add('entering');
    setTimeout(() => wrapper.classList.remove('entering'), 500);
    
    // Atualizar progresso
    progressCount.textContent = `${index + 1}/3`;
    progressFill.style.width = `${((index + 1) / 3) * 100}%`;
    
    // Desenhar camada raspável
    setupCanvas(newCtx, newCanvas);
    
    // Configurar eventos de raspar
    let isScratching = false;
    let hasRevealed = false;
    
    function scratch(x, y) {
        if (hasRevealed) return;
        
        newCtx.globalCompositeOperation = 'destination-out';
        newCtx.beginPath();
        newCtx.arc(x, y, 25, 0, Math.PI * 2);
        newCtx.fill();
        
        const progress = calculateScratchProgress(newCtx, newCanvas);
        
        if (progress >= REVEAL_THRESHOLD && !hasRevealed) {
            hasRevealed = true;
            revealPrize(index);
        }
    }
    
    // Mouse events
    newCanvas.addEventListener('mousedown', (e) => {
        isScratching = true;
        const rect = newCanvas.getBoundingClientRect();
        const x = (e.clientX - rect.left) * (newCanvas.width / rect.width);
        const y = (e.clientY - rect.top) * (newCanvas.height / rect.height);
        scratch(x, y);
    });
    
    newCanvas.addEventListener('mousemove', (e) => {
        if (!isScratching) return;
        const rect = newCanvas.getBoundingClientRect();
        const x = (e.clientX - rect.left) * (newCanvas.width / rect.width);
        const y = (e.clientY - rect.top) * (newCanvas.height / rect.height);
        scratch(x, y);
    });
    
    newCanvas.addEventListener('mouseup', () => { isScratching = false; });
    newCanvas.addEventListener('mouseleave', () => { isScratching = false; });
    
    // Touch events
    newCanvas.addEventListener('touchstart', (e) => {
        e.preventDefault();
        isScratching = true;
        const rect = newCanvas.getBoundingClientRect();
        const touch = e.touches[0];
        const x = (touch.clientX - rect.left) * (newCanvas.width / rect.width);
        const y = (touch.clientY - rect.top) * (newCanvas.height / rect.height);
        scratch(x, y);
    });
    
    newCanvas.addEventListener('touchmove', (e) => {
        e.preventDefault();
        if (!isScratching) return;
        const rect = newCanvas.getBoundingClientRect();
        const touch = e.touches[0];
        const x = (touch.clientX - rect.left) * (newCanvas.width / rect.width);
        const y = (touch.clientY - rect.top) * (newCanvas.height / rect.height);
        scratch(x, y);
    });
    
    newCanvas.addEventListener('touchend', (e) => {
        e.preventDefault();
        isScratching = false;
    });
}

function setupCanvas(ctx, canvas) {
    // Gradiente prateado
    const gradient = ctx.createLinearGradient(0, 0, canvas.width, canvas.height);
    gradient.addColorStop(0, '#c0c0c0');
    gradient.addColorStop(0.5, '#e8e8e8');
    gradient.addColorStop(1, '#a8a8a8');
    
    ctx.fillStyle = gradient;
    ctx.fillRect(0, 0, canvas.width, canvas.height);
    
    // Texto "RASPE AQUI"
    ctx.fillStyle = 'rgba(0, 0, 0, 0.3)';
    ctx.font = 'bold 24px Outfit';
    ctx.textAlign = 'center';
    ctx.textBaseline = 'middle';
    ctx.fillText('RASPE AQUI', canvas.width / 2, canvas.height / 2);
    
    // Emoji de mão
    ctx.font = '48px Arial';
    ctx.fillText('👆', canvas.width / 2, canvas.height / 2 - 60);
}

function calculateScratchProgress(ctx, canvas) {
    const imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);
    const pixels = imageData.data;
    let transparent = 0;
    
    for (let i = 3; i < pixels.length; i += 4) {
        if (pixels[i] < 128) {
            transparent++;
        }
    }
    
    return (transparent / (pixels.length / 4)) * 100;
}

function revealPrize(index) {
    const canvas = document.getElementById('scratchCanvas');
    const scratchContent = document.getElementById('scratchContent');
    const prizeIcon = document.getElementById('prizeIcon');
    const prizeText = document.getElementById('prizeText');
    const instructionText = document.getElementById('instructionText');
    
    const card = SCRATCH_CARDS[index];
    
    // Esconder canvas
    canvas.style.opacity = '0';
    canvas.style.transition = 'opacity 0.3s ease';
    
    setTimeout(() => {
        canvas.style.display = 'none';
        scratchContent.style.display = 'flex';
        
        // Mostrar prêmio
        if (card.type === 'loss') {
            // Não ganhou
            prizeIcon.innerHTML = `
                <div class="prize-icon no-prize">
                    <div class="sad-emoji">😢</div>
                </div>
            `;
            prizeText.innerHTML = `
                <h2>😢 NÃO FOI DESSA VEZ!</h2>
                <p class="prize-subtitle">Continue tentando</p>
            `;
        } else {
            // Ganhou
            totalPrizeValue += card.prize;
            prizeIcon.innerHTML = `
                <img src="../assets/gif.gif" alt="Prêmio" class="prize-gif" style="width: 120px; height: 120px; border-radius: 50%; filter: drop-shadow(0 0 20px rgba(255, 215, 0, 0.5));">
            `;
            prizeText.innerHTML = `
                <h2>🎉 GANHOU!</h2>
                <p class="prize-amount">R$ ${card.prize.toFixed(2)}</p>
                <p class="prize-subtitle">Créditos</p>
            `;
        }
        
        // Próxima ação
        currentCardIndex++;
        
        if (currentCardIndex < SCRATCH_CARDS.length) {
            // Ainda tem mais cards
            instructionText.innerHTML = `✨ Continue raspando! (${currentCardIndex}/3)`;
            
            setTimeout(() => {
                loadCard(currentCardIndex);
            }, 2500); // 2.5s para ver o resultado
        } else {
            // Terminou todas
            instructionText.innerHTML = '🎊 Você completou todas as raspadinhas!';
            showFinalButton();
        }
    }, 300);
}

function showFinalButton() {
    const actionButton = document.getElementById('actionButtonContainer');
    const claimButton = document.getElementById('claimButton');
    const buttonText = claimButton.querySelector('.button-text');
    
    if (totalPrizeValue > 0) {
        buttonText.textContent = `RESGATAR R$ ${totalPrizeValue.toFixed(2)} E JOGAR`;
    } else {
        buttonText.textContent = `JOGAR AGORA`;
    }
    
    actionButton.style.display = 'block';
}

// Botão de claim
document.addEventListener('DOMContentLoaded', () => {
    const claimButton = document.getElementById('claimButton');
    
    claimButton.addEventListener('click', () => {
        const redirectURL = buildURLWithUTMs('/teste/');
        window.location.href = redirectURL;
    });
    
    // Legal footer toggle
    const legalToggle = document.getElementById('legalToggle');
    const legalFooter = document.getElementById('legalFooter');
    
    if (legalToggle && legalFooter) {
        legalToggle.addEventListener('click', () => {
            legalFooter.classList.toggle('open');
        });
    }
});
