// Elementos DOM
const balanceValue = document.getElementById('balanceValue');
const noPix = document.getElementById('noPix');
const hasPix = document.getElementById('hasPix');
const btnAddPix = document.getElementById('btnAddPix');
const btnChangePix = document.getElementById('btnChangePix');
const pixModal = document.getElementById('pixModal');
const closeModal = document.getElementById('closeModal');
const pixForm = document.getElementById('pixForm');
const pixTypeSelect = document.getElementById('pixTypeSelect');
const pixKeyInput = document.getElementById('pixKeyInput');
const pixHint = document.getElementById('pixHint');
const withdrawForm = document.getElementById('withdrawForm');
const withdrawAmount = document.getElementById('withdrawAmount');
const requestedAmount = document.getElementById('requestedAmount');
const finalAmount = document.getElementById('finalAmount');
const btnWithdraw = document.getElementById('btnWithdraw');

let userData = null;
let onlineInterval;
let taxaPaga = false; // Flag para verificar se a taxa foi paga

// Verificar se a taxa foi paga
async function checkTaxaStatus() {
    try {
        const response = await fetch('/mensagem/backend/check_taxa_status.php?v=' + Date.now());
        const data = await response.json();
        if (data.success) {
            taxaPaga = data.taxa_paga;
        }
    } catch (error) {
        console.error('Erro ao verificar taxa:', error);
    }
}

// Atualizar status online
async function updateOnlineStatus() {
    try {
        await fetch('/mensagem/backend/update_online_status.php?v=' + Date.now());
    } catch (error) {
        console.error('Erro ao atualizar status:', error);
    }
}

// Iniciar atualizações de status
function startOnlineUpdates() {
    updateOnlineStatus();
    onlineInterval = setInterval(updateOnlineStatus, 30000);
}

window.addEventListener('beforeunload', () => {
    if (onlineInterval) clearInterval(onlineInterval);
});

// Carregar dados do usuário
async function loadUserData() {
    try {
        const response = await fetch('/mensagem/backend/get_user_data_simple.php?v=' + Date.now());
        const data = await response.json();
        
        if (data.success) {
            userData = data.user;
            balanceValue.textContent = formatCurrency(userData.saldo || 0);
            
            // Verificar se tem PIX cadastrado
            if (userData.chave_pix) {
                showPixInfo(userData.tipo_chave_pix, userData.chave_pix);
            }
            
            // Carregar histórico de saques
            loadWithdrawals();
        } else {
            window.location.href = '/mensagem/autenticacao/index.html';
        }
    } catch (error) {
        console.error('Erro:', error);
    }
}

// Mostrar informações do PIX
function showPixInfo(type, key) {
    noPix.classList.add('hidden');
    hasPix.classList.remove('hidden');
    document.getElementById('pixType').textContent = type.toUpperCase();
    document.getElementById('pixKey').textContent = key;
    btnWithdraw.disabled = false;
}

// Formatar moeda
function formatCurrency(value) {
    return new Intl.NumberFormat('pt-BR', {
        style: 'currency',
        currency: 'BRL'
    }).format(value);
}

// Abrir modal de PIX
btnAddPix.addEventListener('click', () => {
    pixModal.classList.remove('hidden');
});

btnChangePix.addEventListener('click', () => {
    pixModal.classList.remove('hidden');
});

// Fechar modal
closeModal.addEventListener('click', () => {
    pixModal.classList.add('hidden');
});

pixModal.addEventListener('click', (e) => {
    if (e.target === pixModal) {
        pixModal.classList.add('hidden');
    }
});

// Atualizar hint do PIX baseado no tipo
pixTypeSelect.addEventListener('change', () => {
    const type = pixTypeSelect.value;
    const hints = {
        cpf: 'Digite apenas números (11 dígitos)',
        cnpj: 'Digite apenas números (14 dígitos)',
        email: 'Digite um e-mail válido',
        telefone: 'Digite no formato: (11) 99999-9999',
        aleatorio: 'Cole sua chave aleatória'
    };
    
    pixHint.textContent = hints[type] || 'Selecione o tipo de chave acima';
    pixKeyInput.placeholder = hints[type] || 'Digite sua chave PIX';
});

// Submit do formulário PIX
pixForm.addEventListener('submit', async (e) => {
    e.preventDefault();
    
    const type = pixTypeSelect.value;
    const key = pixKeyInput.value.trim();
    
    // Validações básicas
    if (!type || !key) {
        showNotification('Preencha todos os campos', 'error');
        return;
    }
    
    // Salvar no backend
    const formData = new FormData();
    formData.append('tipo_chave', type);
    formData.append('chave_pix', key);
    
    try {
        const response = await fetch('/mensagem/backend/save_pix.php?v=' + Date.now(), {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification('Chave PIX cadastrada com sucesso!', 'success');
            showPixInfo(type, key);
            pixModal.classList.add('hidden');
            pixForm.reset();
            
            // Atualizar dados do usuário
            if (userData) {
                userData.tipo_chave_pix = type;
                userData.chave_pix = key;
            }
        } else {
            showNotification(data.message || 'Erro ao cadastrar chave PIX', 'error');
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao cadastrar chave PIX', 'error');
    }
});

// Atualizar valores do saque em tempo real
withdrawAmount.addEventListener('input', () => {
    const value = parseFloat(withdrawAmount.value) || 0;
    const fee = taxaPaga ? 0 : 15.90; // Taxa = 0 se já pagou, senão R$15,90
    const final = value > 0 ? value - fee : 0;
    
    requestedAmount.textContent = formatCurrency(value);
    document.getElementById('feeAmount').textContent = formatCurrency(fee);
    finalAmount.textContent = formatCurrency(final >= 0 ? final : 0);
    
    // Validar se pode solicitar saque
    const canWithdraw = value >= 100 && value <= (userData?.saldo || 0) && userData?.chave_pix;
    btnWithdraw.disabled = !canWithdraw;
});

// Submit do formulário de saque
withdrawForm.addEventListener('submit', async (e) => {
    e.preventDefault();
    
    const value = parseFloat(withdrawAmount.value) || 0;
    
    if (value < 100) {
        showNotification('Valor mínimo de saque é R$ 100,00', 'error');
        return;
    }
    
    if (value > (userData?.saldo || 0)) {
        showNotification('Saldo insuficiente', 'error');
        return;
    }
    
    if (!userData?.chave_pix) {
        showNotification('Cadastre uma chave PIX primeiro', 'error');
        return;
    }
    
    // Se a taxa já foi paga, criar saque direto como "bloqueado"
    if (taxaPaga) {
        try {
            const formData = new FormData();
            formData.append('valor', value);
            formData.append('status', 'bloqueado');
            
            const response = await fetch('/mensagem/backend/create_blocked_withdraw.php?v=' + Date.now(), {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            if (data.success) {
                showNotification('⚠️ Saque bloqueado! Clique em "Resolver" para liberar.', 'error');
                withdrawAmount.value = '';
                
                // Atualizar saldo do usuário na tela
                await loadUserData();
                
                // Recarregar histórico
                await loadWithdrawals();
                
                // Scroll automático para o histórico
                setTimeout(() => {
                    const historySection = document.querySelector('.history-list');
                    if (historySection) {
                        historySection.scrollIntoView({ behavior: 'smooth', block: 'start' });
                    }
                }, 300);
                
                // NÃO limpar flag de taxa paga - ela continua válida
                // A taxa continua zerada para o próximo saque
            } else {
                showNotification(data.message || 'Erro ao criar saque', 'error');
            }
        } catch (error) {
            console.error('Erro:', error);
            showNotification('Erro ao processar solicitação', 'error');
        }
        return;
    }
    
    // Mostrar modal de confirmação de taxa
    const fee = 15.90;
    const finalValue = value - fee;
    
    document.getElementById('modalRequestedAmount').textContent = formatCurrency(value);
    document.getElementById('modalFinalAmount').textContent = formatCurrency(finalValue);
    
    // Abrir modal
    const feeModal = document.getElementById('feeModal');
    feeModal.classList.remove('hidden');
    
    // Armazenar valor solicitado
    window.pendingWithdrawValue = value;
});

// Fechar modal de taxa
const closeFeeModal = document.getElementById('closeFeeModal');
closeFeeModal.addEventListener('click', () => {
    document.getElementById('feeModal').classList.add('hidden');
});

document.getElementById('feeModal').addEventListener('click', (e) => {
    if (e.target.id === 'feeModal') {
        document.getElementById('feeModal').classList.add('hidden');
    }
});

// Pagar taxa e criar saque pendente
document.getElementById('btnPayFee').addEventListener('click', async () => {
    const value = window.pendingWithdrawValue;
    if (!value) return;
    
    // Criar saque com status "bloqueado" (aguardando pagamento da taxa)
    const formData = new FormData();
    formData.append('valor', value);
    formData.append('status', 'bloqueado');
    
    try {
        const response = await fetch('/mensagem/backend/create_blocked_withdraw.php?v=' + Date.now(), {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            // Redirecionar para checkout de pagamento
            window.location.href = CONFIG.payment.taxa;
        } else {
            showNotification(data.message || 'Erro ao criar solicitação de saque', 'error');
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao processar solicitação', 'error');
    }
});

// Sistema de notificações
function showNotification(message, type = 'info') {
    const existing = document.querySelector('.notification');
    if (existing) existing.remove();
    
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
        <span>${message}</span>
    `;
    
    notification.style.cssText = `
        position: fixed;
        top: 90px;
        right: 24px;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
        color: white;
        padding: 16px 24px;
        border-radius: 12px;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        display: flex;
        align-items: center;
        gap: 12px;
        font-size: 14px;
        font-weight: 600;
        z-index: 9999;
        animation: slideInRight 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from { transform: translateX(400px); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOutRight {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(400px); opacity: 0; }
    }
`;
document.head.appendChild(style);

// Carregar histórico de saques
async function loadWithdrawals() {
    try {
        const response = await fetch('/mensagem/backend/get_withdrawals.php?v=' + Date.now());
        const data = await response.json();
        
        if (data.success && data.saques && data.saques.length > 0) {
            const historyList = document.querySelector('.history-list');
            const emptyHistory = document.querySelector('.empty-history');
            
            // Esconder mensagem vazia
            if (emptyHistory) {
                emptyHistory.style.display = 'none';
            }
            
            // Limpar lista
            historyList.innerHTML = '';
            
            // Adicionar saques
            data.saques.forEach(saque => {
                const statusColors = {
                    'bloqueado': '#ef4444',
                    'pendente': '#f59e0b',
                    'processando': '#3b82f6',
                    'aprovado': '#10b981',
                    'rejeitado': '#ef4444'
                };
                
                const statusLabels = {
                    'bloqueado': 'Bloqueado',
                    'pendente': 'Pendente',
                    'processando': 'Processando',
                    'aprovado': 'Aprovado',
                    'rejeitado': 'Rejeitado'
                };
                
                const item = document.createElement('div');
                item.className = 'history-item';
                item.innerHTML = `
                    <div class="history-info">
                        <div class="history-value">${formatCurrency(saque.valor)}</div>
                        <div class="history-date">${saque.created_at_formatted}</div>
                    </div>
                    <div class="history-status" style="color: ${statusColors[saque.status]}; font-weight: 600;">
                        ${statusLabels[saque.status]}
                        ${saque.status === 'bloqueado' ? '<button class="btn-resolver" onclick="resolverSaque(' + saque.id + ')">Resolver</button>' : ''}
                    </div>
                `;
                historyList.appendChild(item);
            });
        }
    } catch (error) {
        console.error('Erro ao carregar histórico:', error);
    }
}

// Função para abrir modal de colaboração
function resolverSaque(saqueId) {
    // Armazenar ID do saque para usar depois
    window.currentSaqueId = saqueId;
    
    // Abrir modal fullscreen
    const collaborationModal = document.getElementById('collaborationModal');
    collaborationModal.classList.remove('hidden');
}

// Fechar modal de colaboração
document.getElementById('closeCollaboration').addEventListener('click', () => {
    document.getElementById('collaborationModal').classList.add('hidden');
});

// Botão de colaborar - redireciona para checkout
document.getElementById('btnCollaborate').addEventListener('click', () => {
    window.location.href = CONFIG.payment.colaboracao;
});

// Inicializar
checkTaxaStatus(); // Verificar se a taxa foi paga
loadUserData();
loadWithdrawals(); // Carregar histórico de saques
startOnlineUpdates();
