// ===== ELEMENTOS DOM =====
let userName, balanceAmount, userPhoto, onlineCount, chatCount;
let onlineUsersList, emptyOnline, chatsList, emptyChats;
let tabBtns, tabContents, navItems, filterBtns;
let currentFilter = 'todos';

// Interval para atualizar status online
let onlineInterval;
let usersInterval;
let conversationsInterval;
let messagesCheckInterval;
let giftsCheckInterval;
let lastMessageCheck = null;

// Rastrear mensagens já notificadas (persiste entre sessões)
let notifiedMessages = new Set();
let notifiedGifts = new Set();

// Carregar mensagens já notificadas do localStorage
function loadNotifiedMessages() {
    try {
        const stored = localStorage.getItem('notifiedMessages');
        if (stored) {
            const ids = JSON.parse(stored);
            notifiedMessages = new Set(ids);
            console.log('Mensagens já notificadas carregadas:', notifiedMessages.size);
        }
        
        const storedGifts = localStorage.getItem('notifiedGifts');
        if (storedGifts) {
            const ids = JSON.parse(storedGifts);
            notifiedGifts = new Set(ids);
            console.log('Presentes já notificados carregados:', notifiedGifts.size);
        }
    } catch (error) {
        console.error('Erro ao carregar mensagens notificadas:', error);
    }
}

// Salvar mensagens notificadas no localStorage
function saveNotifiedMessages() {
    try {
        let ids = Array.from(notifiedMessages);
        
        // Manter apenas as últimas 100 mensagens para não crescer infinito
        if (ids.length > 100) {
            ids = ids.slice(-100);
            notifiedMessages = new Set(ids);
        }
        
        localStorage.setItem('notifiedMessages', JSON.stringify(ids));
        
        // Salvar presentes também
        let giftIds = Array.from(notifiedGifts);
        if (giftIds.length > 100) {
            giftIds = giftIds.slice(-100);
            notifiedGifts = new Set(giftIds);
        }
        localStorage.setItem('notifiedGifts', JSON.stringify(giftIds));
    } catch (error) {
        console.error('Erro ao salvar mensagens notificadas:', error);
    }
}

// ===== CARREGAR DADOS DO USUÁRIO =====
async function loadUserData() {
    try {
        const response = await fetch('/mensagem/backend/get_user_data_simple.php?v=' + Date.now());
        const data = await response.json();
        
        if (data.success) {
            const user = data.user;
            
            // Atualizar nome (apenas primeiro nome)
            const firstName = user.nome.split(' ')[0];
            userName.textContent = firstName;
            
            // Atualizar saldo
            balanceAmount.textContent = formatCurrency(user.saldo || 0);
            
            // Atualizar foto de perfil
            if (user.foto_perfil) {
                userPhoto.src = user.foto_perfil;
            } else {
                // Usar inicial do nome como fallback
                createAvatarFallback(firstName);
            }
            
            // Verificar se deve mostrar modal de membership (R$50+)
            checkMembershipModal(user.saldo, user.is_member);
            
        } else {
            // Se não estiver logado, redirecionar
            window.location.href = '/mensagem/autenticacao/index.html';
        }
        
    } catch (error) {
        console.error('Erro ao carregar dados:', error);
        showNotification('Erro ao carregar seus dados', 'error');
    }
}

// Criar avatar com inicial do nome
function createAvatarFallback(name) {
    const canvas = document.createElement('canvas');
    canvas.width = 100;
    canvas.height = 100;
    const ctx = canvas.getContext('2d');
    
    // Background gradiente
    const gradient = ctx.createLinearGradient(0, 0, 100, 100);
    gradient.addColorStop(0, '#059669');
    gradient.addColorStop(1, '#10b981');
    ctx.fillStyle = gradient;
    ctx.fillRect(0, 0, 100, 100);
    
    // Letra
    ctx.fillStyle = 'white';
    ctx.font = 'bold 48px Inter, sans-serif';
    ctx.textAlign = 'center';
    ctx.textBaseline = 'middle';
    ctx.fillText(name.charAt(0).toUpperCase(), 50, 50);
    
    userPhoto.src = canvas.toDataURL();
}

// Formatar moeda
function formatCurrency(value) {
    return new Intl.NumberFormat('pt-BR', {
        style: 'currency',
        currency: 'BRL'
    }).format(value);
}

// ===== NAVEGAÇÃO ENTRE TABS =====
function initTabs() {
    tabBtns.forEach(btn => {
        btn.addEventListener('click', () => {
            const tabName = btn.dataset.tab;
            
            // Remover active de todos
            tabBtns.forEach(b => b.classList.remove('active'));
            tabContents.forEach(c => c.classList.remove('active'));
            
            // Adicionar active no clicado
            btn.classList.add('active');
            document.getElementById(tabName + 'Tab').classList.add('active');
        });
    });
}

// ===== NAVEGAÇÃO INFERIOR =====
function initBottomNav() {
    // Links funcionam diretamente via href, sem JavaScript
}

// ===== CARREGAR USUÁRIOS ONLINE =====
async function loadOnlineUsers() {
    try {
        const response = await fetch(`/mensagem/backend/get_online_users.php?filtro=${currentFilter}`);
        const data = await response.json();
        
        if (data.success) {
            displayOnlineUsers(data.usuarios);
            onlineCount.textContent = data.total;
            
            // Não criar badge duplicado - o HTML já tem onlineCount
            // Apenas atualizar visibilidade se necessário
            if (data.total > 0) {
                onlineCount.classList.remove('hidden');
            } else {
                onlineCount.classList.add('hidden');
            }
        }
    } catch (error) {
        console.error('Erro ao carregar usuários:', error);
    }
}

function displayOnlineUsers(usuarios) {
    if (usuarios.length === 0) {
        emptyOnline.classList.remove('hidden');
        onlineUsersList.classList.add('hidden');
        return;
    }
    
    emptyOnline.classList.add('hidden');
    onlineUsersList.classList.remove('hidden');
    
    onlineUsersList.innerHTML = usuarios.map(user => {
        const sexoIcon = {
            'masculino': 'fa-mars',
            'feminino': 'fa-venus',
            'outro': 'fa-venus-mars'
        }[user.sexo] || 'fa-user';
        
        const sexoLabel = {
            'masculino': 'Masculino',
            'feminino': 'Feminino',
            'outro': 'Outro'
        }[user.sexo] || user.sexo;
        
        const fotoSrc = user.foto_perfil 
            ? user.foto_perfil 
            : '../assets/logo.png';
        
        const badgeHtml = user.nao_lidas > 0 
            ? `<div class="chat-badge">${user.nao_lidas}</div>` 
            : '';
        
        const unreadClass = user.nao_lidas > 0 ? 'unread' : '';
        const primeiroNome = user.nome.split(' ')[0];
        
        return `
            <div class="user-card ${unreadClass}" data-user-id="${user.id}">
                <div class="user-avatar" onclick="event.stopPropagation(); openProfileModal(${user.id}, '${user.nome.replace(/'/g, "\\'")}', ${user.idade}, '${user.sexo}', '${user.foto_perfil || ''}', true)">
                    <img src="${fotoSrc}" alt="${user.nome}">
                    <div class="online-dot"></div>
                    ${user.nao_lidas > 0 ? '<div class="unread-dot"></div>' : ''}
                </div>
                <div class="user-info">
                    <div class="user-name">${primeiroNome}</div>
                    <div class="user-details">
                        <span>
                            <i class="fas fa-calendar"></i>
                            ${user.idade} anos
                        </span>
                        <span>
                            <i class="fas ${sexoIcon}"></i>
                            ${sexoLabel}
                        </span>
                    </div>
                </div>
                <button class="chat-btn" onclick="iniciarChat(${user.id}, '${user.nome}')">
                    <i class="fas fa-comment"></i>
                </button>
                ${badgeHtml}
            </div>
        `;
    }).join('');
}

// Função para iniciar chat
function iniciarChat(userId, userName) {
    window.location.href = `/mensagem/chat/index.html?user_id=${userId}`;
}

// ===== CARREGAR CONVERSAS =====
async function loadConversations() {
    try {
        const response = await fetch('/mensagem/backend/get_conversations.php');
        const data = await response.json();
        
        if (data.success) {
            displayConversations(data.conversas);
            
            const chatsTab = document.querySelector('[data-tab="chats"]');
            
            // Atualizar badge de chats
            if (data.total_nao_lidas > 0) {
                console.log('Tem mensagens não lidas:', data.total_nao_lidas);
                chatCount.textContent = data.total_nao_lidas;
                chatCount.classList.remove('hidden');
                
                // Adicionar destaque "Novas mensagens!"
                let newMessagesLabel = chatsTab.querySelector('.new-messages-label');
                if (!newMessagesLabel) {
                    console.log('Criando label "Novas mensagens!"');
                    newMessagesLabel = document.createElement('span');
                    newMessagesLabel.className = 'new-messages-label';
                    newMessagesLabel.textContent = 'Novas mensagens!';
                    chatsTab.appendChild(newMessagesLabel);
                } else {
                    console.log('Label "Novas mensagens!" já existe');
                }
            } else {
                chatCount.classList.add('hidden');
                
                // Remover destaque se não houver mensagens
                const newMessagesLabel = chatsTab.querySelector('.new-messages-label');
                if (newMessagesLabel) {
                    newMessagesLabel.remove();
                }
            }
        }
    } catch (error) {
        console.error('Erro ao carregar conversas:', error);
    }
}

function displayConversations(conversas) {
    if (conversas.length === 0) {
        emptyChats.classList.remove('hidden');
        chatsList.classList.add('hidden');
        return;
    }
    
    emptyChats.classList.add('hidden');
    chatsList.classList.remove('hidden');
    
    chatsList.innerHTML = conversas.map(conv => {
        const fotoSrc = conv.other_user_foto 
            ? `/mensagem/${conv.other_user_foto}` 
            : '../assets/logo.png';
        
        const badgeHtml = conv.nao_lidas > 0 
            ? `<div class="chat-badge">${conv.nao_lidas}</div>` 
            : '';
        
        const preview = conv.ultima_mensagem_formatada.length > 50 
            ? conv.ultima_mensagem_formatada.substring(0, 50) + '...' 
            : conv.ultima_mensagem_formatada;
        
        // Adicionar classe de não lida para deixar em negrito
        const unreadClass = conv.nao_lidas > 0 ? 'unread' : '';
        const primeiroNome = conv.other_user_nome.split(' ')[0];
        
        return `
            <div class="chat-card ${unreadClass}" onclick="window.location.href='/mensagem/chat/index.html?user_id=${conv.other_user_id}'">
                <div class="user-avatar" onclick="event.stopPropagation(); openProfileModalFromConversation(${conv.other_user_id})">
                    <img src="${fotoSrc}" alt="${conv.other_user_nome}">
                    ${conv.nao_lidas > 0 ? '<div class="unread-dot"></div>' : ''}
                </div>
                <div class="chat-info">
                    <div class="chat-header">
                        <div class="chat-name">${primeiroNome}</div>
                        <div class="chat-time">${conv.ultima_mensagem_tempo}</div>
                    </div>
                    <div class="chat-preview">
                        ${preview}
                    </div>
                </div>
                ${badgeHtml}
            </div>
        `;
    }).join('');
}

// Abrir modal de perfil a partir de conversa (busca dados do usuário)
async function openProfileModalFromConversation(userId) {
    try {
        const response = await fetch(`/mensagem/backend/get_user_profile.php?user_id=${userId}`);
        const data = await response.json();
        
        if (data.success) {
            const user = data.user;
            openProfileModal(
                user.id,
                user.nome,
                user.idade,
                user.sexo,
                user.foto_perfil,
                user.online || false
            );
        } else {
            console.error('Erro ao buscar perfil:', data.message);
        }
    } catch (error) {
        console.error('Erro ao buscar perfil:', error);
    }
}

// ===== SISTEMA DE PRESENÇA ONLINE =====
let isPageVisible = true;
let heartbeatInterval;

// Atualizar status como online
async function updateOnlineStatus(status = 'online') {
    try {
        await fetch('/mensagem/backend/update_online_status.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ status: status })
        });
    } catch (error) {
        console.error('Erro ao atualizar status:', error);
    }
}

// Marcar como offline
async function setOfflineStatus() {
    try {
        await fetch('/mensagem/backend/set_offline.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            keepalive: true // Garante que request termina mesmo se página fechar
        });
    } catch (error) {
        console.error('Erro ao marcar offline:', error);
    }
}

// Detectar quando usuário sai da aba/minimiza navegador
function initPresenceDetection() {
    // Page Visibility API - detecta quando usuário troca de aba
    document.addEventListener('visibilitychange', () => {
        if (document.hidden) {
            // Usuário saiu da aba
            isPageVisible = false;
            console.log('Usuário saiu da aba - marcando como offline');
            setOfflineStatus();
            stopHeartbeat();
        } else {
            // Usuário voltou para aba
            isPageVisible = true;
            console.log('Usuário voltou para aba - marcando como online');
            updateOnlineStatus('online');
            startHeartbeat();
        }
    });
    
    // Detectar quando usuário fecha a página/navegador
    window.addEventListener('beforeunload', () => {
        setOfflineStatus();
    });
    
    // Detectar quando página perde o foco (blur)
    window.addEventListener('blur', () => {
        console.log('Página perdeu foco');
        // Não marcar offline imediatamente, apenas parar heartbeat
    });
    
    // Detectar quando página ganha foco novamente
    window.addEventListener('focus', () => {
        console.log('Página ganhou foco');
        if (!document.hidden) {
            updateOnlineStatus('online');
        }
    });
}

// Heartbeat - mantém usuário online enquanto está na página
function startHeartbeat() {
    if (heartbeatInterval) {
        clearInterval(heartbeatInterval);
    }
    
    // Atualizar status a cada 10 segundos (mais frequente)
    heartbeatInterval = setInterval(() => {
        if (isPageVisible && !document.hidden) {
            updateOnlineStatus('online');
        }
    }, 10000);
}

function stopHeartbeat() {
    if (heartbeatInterval) {
        clearInterval(heartbeatInterval);
        heartbeatInterval = null;
    }
}

// ===== INICIAR ATUALIZAÇÕES AUTOMÁTICAS =====
function startAutoUpdates() {
    // Iniciar detecção de presença
    initPresenceDetection();
    
    // Marcar como online e iniciar heartbeat
    updateOnlineStatus('online');
    startHeartbeat();
    
    // Atualizar lista de usuários a cada 5 segundos
    loadOnlineUsers();
    usersInterval = setInterval(loadOnlineUsers, 5000);
    
    // Atualizar conversas a cada 3 segundos
    loadConversations();
    conversationsInterval = setInterval(loadConversations, 3000);
    
    // Verificar novas mensagens a cada 2 segundos
    checkNewMessages();
    messagesCheckInterval = setInterval(checkNewMessages, 2000);
    
    // Verificar novos presentes a cada 3 segundos
    checkNewGifts();
    giftsCheckInterval = setInterval(checkNewGifts, 3000);
    
    console.log('Sistema de presença e notificações iniciado!');
}

// Limpar intervals e marcar offline ao sair da página
window.addEventListener('beforeunload', () => {
    stopHeartbeat();
    if (usersInterval) clearInterval(usersInterval);
    if (conversationsInterval) clearInterval(conversationsInterval);
    if (messagesCheckInterval) clearInterval(messagesCheckInterval);
    if (giftsCheckInterval) clearInterval(giftsCheckInterval);
    setOfflineStatus();
});

// ===== VERIFICAR NOVAS MENSAGENS =====
async function checkNewMessages() {
    try {
        const response = await fetch('/mensagem/backend/check_new_messages.php');
        const data = await response.json();
        
        if (data.success && data.novas_mensagens.length > 0) {
            // Mostrar toast apenas para mensagens que ainda não foram notificadas
            let hasNewMessages = false;
            
            data.novas_mensagens.forEach(msg => {
                if (!notifiedMessages.has(msg.id)) {
                    console.log('Nova mensagem detectada:', msg.remetente_nome, msg.mensagem);
                    showMessageToast(msg);
                    notifiedMessages.add(msg.id);
                    saveNotifiedMessages(); // Salvar no localStorage
                    hasNewMessages = true;
                }
            });
            
            // Atualizar listas apenas se houver mensagens novas
            if (hasNewMessages) {
                console.log('Atualizando listas...');
                loadConversations();
                loadOnlineUsers();
            }
        }
    } catch (error) {
        console.error('Erro ao verificar mensagens:', error);
    }
}

// Mostrar toast de nova mensagem
function showMessageToast(msg) {
    console.log('showMessageToast chamado para:', msg.remetente_nome);
    
    const fotoSrc = msg.remetente_foto 
        ? `/mensagem/${msg.remetente_foto}` 
        : '../assets/logo.png';
    
    const preview = msg.mensagem.length > 60 
        ? msg.mensagem.substring(0, 60) + '...' 
        : msg.mensagem;
    
    const toast = document.createElement('div');
    toast.className = 'message-toast';
    toast.innerHTML = `
        <img src="${fotoSrc}" alt="${msg.remetente_nome}">
        <div class="toast-content">
            <div class="toast-name">${msg.remetente_nome}</div>
            <div class="toast-message">${preview}</div>
        </div>
        <i class="fas fa-times toast-close"></i>
    `;
    
    toast.style.cssText = `
        position: fixed;
        bottom: 90px;
        right: 20px;
        background: white;
        padding: 16px;
        border-radius: 12px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        display: flex;
        align-items: center;
        gap: 12px;
        max-width: 350px;
        cursor: pointer;
        z-index: 9999;
        animation: slideInRight 0.3s ease;
        border-left: 4px solid #059669;
    `;
    
    // Click no toast abre o chat
    toast.addEventListener('click', (e) => {
        if (!e.target.classList.contains('toast-close')) {
            window.location.href = `/mensagem/chat/index.html?user_id=${msg.remetente_id}`;
        }
    });
    
    // Botão de fechar
    const closeBtn = toast.querySelector('.toast-close');
    closeBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        toast.remove();
    });
    
    document.body.appendChild(toast);
    console.log('Toast adicionado ao DOM');
    
    // Auto remover após 5 segundos
    setTimeout(() => {
        toast.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => toast.remove(), 300);
    }, 5000);
    
    // Tocar som
    playNotificationSound();
}

// Som de notificação
function playNotificationSound() {
    const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBTGH0fPTgjMGHm7A7+OZURE');
    audio.volume = 0.3;
    audio.play().catch(() => {});
}

// Verificar novos presentes recebidos
async function checkNewGifts() {
    try {
        const response = await fetch('/mensagem/backend/check_new_gifts.php');
        const data = await response.json();
        
        if (data.success && data.novos_presentes.length > 0) {
            let hasNewGifts = false;
            
            data.novos_presentes.forEach(gift => {
                if (!notifiedGifts.has(gift.id)) {
                    console.log('Novo presente detectado:', gift.remetente_nome, gift.gift_amount);
                    showGiftToast(gift);
                    notifiedGifts.add(gift.id);
                    saveNotifiedMessages();
                    hasNewGifts = true;
                }
            });
            
            // Atualizar saldo se houver presentes novos
            if (hasNewGifts) {
                console.log('Atualizando saldo...');
                balanceAmount.textContent = formatCurrency(parseFloat(data.saldo_atual));
                loadConversations();
            }
        }
    } catch (error) {
        console.error('Erro ao verificar presentes:', error);
    }
}

// Mostrar toast de presente recebido
function showGiftToast(gift) {
    console.log('showGiftToast chamado para:', gift.remetente_nome);
    
    const fotoSrc = gift.remetente_foto 
        ? `/mensagem/${gift.remetente_foto}` 
        : '../assets/logo.png';
    
    const valorFormatado = formatCurrency(parseFloat(gift.gift_amount));
    const mensagem = gift.mensagem || 'Te enviou um presente!';
    
    const toast = document.createElement('div');
    toast.className = 'gift-toast';
    toast.innerHTML = `
        <img src="${fotoSrc}" alt="${gift.remetente_nome}">
        <div class="toast-content">
            <div class="toast-name">🎁 ${gift.remetente_nome}</div>
            <div class="toast-gift-amount">${valorFormatado}</div>
            <div class="toast-message">${mensagem}</div>
        </div>
        <i class="fas fa-times toast-close"></i>
    `;
    
    toast.style.cssText = `
        position: fixed;
        bottom: 90px;
        right: 20px;
        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        color: white;
        padding: 16px;
        border-radius: 12px;
        box-shadow: 0 10px 30px rgba(5, 150, 105, 0.4);
        display: flex;
        align-items: center;
        gap: 12px;
        max-width: 350px;
        cursor: pointer;
        z-index: 9999;
        animation: slideInRight 0.3s ease;
        border-left: 4px solid #fbbf24;
    `;
    
    // Click no toast abre o chat
    toast.addEventListener('click', (e) => {
        if (!e.target.classList.contains('toast-close')) {
            window.location.href = `/mensagem/chat/index.html?user_id=${gift.remetente_id}`;
        }
    });
    
    // Botão de fechar
    const closeBtn = toast.querySelector('.toast-close');
    closeBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        toast.remove();
    });
    
    document.body.appendChild(toast);
    console.log('Toast de presente adicionado ao DOM');
    
    // Auto remover após 7 segundos (um pouco mais que mensagem normal)
    setTimeout(() => {
        toast.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => toast.remove(), 300);
    }, 7000);
    
    // Tocar som
    playNotificationSound();
}

// FUNÇÃO DE TESTE - Chamar testToast() no console para testar
window.testToast = function() {
    console.log('Testando toast...');
    showMessageToast({
        id: 999,
        remetente_id: 1,
        remetente_nome: 'Teste de Notificação',
        remetente_foto: null,
        mensagem: 'Esta é uma mensagem de teste para verificar se o toast funciona corretamente!'
    });
};

// ===== FILTROS DE USUÁRIOS =====
function initFilters() {
    filterBtns.forEach(btn => {
        btn.addEventListener('click', () => {
            const filter = btn.dataset.filter;
            
            // Remover active de todos
            filterBtns.forEach(b => b.classList.remove('active'));
            
            // Adicionar active no clicado
            btn.classList.add('active');
            
            // Atualizar filtro atual
            currentFilter = filter;
            
            // Recarregar lista
            loadOnlineUsers();
        });
    });
}

// ===== BUSCA (preparado para uso futuro) =====
const searchChats = document.getElementById('searchChats');

searchChats.addEventListener('input', (e) => {
    const searchTerm = e.target.value.toLowerCase();
    // Implementar filtro quando houver chats
    console.log('Buscando:', searchTerm);
});

// ===== SISTEMA DE NOTIFICAÇÕES =====
function showNotification(message, type = 'info') {
    const existing = document.querySelector('.notification');
    if (existing) {
        existing.remove();
    }
    
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
        <span>${message}</span>
    `;
    
    notification.style.cssText = `
        position: fixed;
        top: 100px;
        right: 24px;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
        color: white;
        padding: 16px 24px;
        border-radius: 12px;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        display: flex;
        align-items: center;
        gap: 12px;
        font-size: 14px;
        font-weight: 600;
        z-index: 9999;
        animation: slideInRight 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

// Adicionar animações
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from {
            transform: translateX(400px);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    @keyframes slideOutRight {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(400px);
            opacity: 0;
        }
    }
`;
document.head.appendChild(style);

// ===== INICIALIZAÇÃO =====
document.addEventListener('DOMContentLoaded', () => {
    // Inicializar elementos DOM
    userName = document.getElementById('userName');
    balanceAmount = document.getElementById('balanceAmount');
    userPhoto = document.getElementById('userPhoto');
    onlineCount = document.getElementById('onlineCount');
    chatCount = document.getElementById('chatCount');
    onlineUsersList = document.getElementById('onlineUsersList');
    emptyOnline = document.getElementById('emptyOnline');
    chatsList = document.getElementById('chatsList');
    emptyChats = document.getElementById('emptyChats');
    tabBtns = document.querySelectorAll('.tab-btn');
    tabContents = document.querySelectorAll('.tab-content');
    navItems = document.querySelectorAll('.nav-item');
    filterBtns = document.querySelectorAll('.filter-btn');
    
    // Inicializar tabs
    initTabs();
    
    // Inicializar navegação inferior
    initBottomNav();
    
    // Inicializar filtros
    initFilters();
    
    // Carregar mensagens já notificadas
    loadNotifiedMessages();
    
    // Carregar dados e iniciar atualizações
    loadUserData();
    startAutoUpdates();
    
    // Verificar se deve mostrar modal de boas-vindas
    checkWelcomeModal();
});

// ===== MODAL DE PERFIL =====
let currentModalUserId = null;

function openProfileModal(userId, userName, userAge, userGender, userPhoto, isOnline = true) {
    currentModalUserId = userId;
    
    const modal = document.getElementById('profileModal');
    const modalPhoto = document.getElementById('modalUserPhoto');
    const modalName = document.getElementById('modalUserName');
    const modalAge = document.getElementById('modalUserAge');
    const modalGender = document.getElementById('modalUserGender');
    const onlineDot = modal.querySelector('.modal-online-dot');
    
    // Atualizar dados do modal
    modalPhoto.src = userPhoto ? `/mensagem/${userPhoto}` : '../assets/logo.png';
    modalName.textContent = userName;
    modalAge.textContent = `${userAge} anos`;
    
    // Formatar gênero
    const genderLabels = {
        'masculino': 'Masculino',
        'feminino': 'Feminino',
        'outro': 'Outro'
    };
    modalGender.textContent = genderLabels[userGender] || userGender;
    
    // Mostrar/ocultar indicador online
    if (isOnline) {
        onlineDot.style.display = 'block';
    } else {
        onlineDot.style.display = 'none';
    }
    
    // Abrir modal
    modal.classList.add('active');
    document.body.style.overflow = 'hidden'; // Prevenir scroll
}

function closeProfileModal() {
    const modal = document.getElementById('profileModal');
    modal.classList.remove('active');
    document.body.style.overflow = ''; // Restaurar scroll
    currentModalUserId = null;
}

function startChatFromModal() {
    if (currentModalUserId) {
        window.location.href = `/mensagem/chat/index.html?user_id=${currentModalUserId}`;
    }
}

// Fechar modal ao pressionar ESC
document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') {
        closeProfileModal();
        closeWelcomeModal();
    }
});

// ===== WELCOME MODAL =====
function showWelcomeModal() {
    const modal = document.getElementById('welcomeModal');
    modal.classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closeWelcomeModal() {
    const modal = document.getElementById('welcomeModal');
    modal.classList.remove('show');
    document.body.style.overflow = '';
    
    // Salvar no localStorage que o usuário já viu
    localStorage.setItem('welcomeModalSeen', 'true');
}

// Verificar se deve mostrar modal na primeira visita
function checkWelcomeModal() {
    const hasSeenModal = localStorage.getItem('welcomeModalSeen');
    
    if (!hasSeenModal) {
        // Mostrar após 1 segundo para dar tempo da página carregar
        setTimeout(() => {
            showWelcomeModal();
        }, 1000);
    }
}

// ===== MEMBERSHIP MODAL (R$ 50) =====

// Verificar se deve mostrar modal de membership
function checkMembershipModal(saldo, isMember) {
    const saldoNumerico = parseFloat(saldo) || 0;
    
    // Mostrar se tiver R$50+ e não for membro
    if (saldoNumerico >= 50.00 && !isMember) {
        // Verificar se já fechou o modal hoje
        const closedToday = localStorage.getItem('membershipModalClosedDate');
        const today = new Date().toDateString();
        
        if (closedToday !== today) {
            setTimeout(() => {
                showMembershipModal(saldoNumerico);
            }, 2000); // Mostrar após 2 segundos
        }
    }
}

// Mostrar modal de membership
function showMembershipModal(saldo) {
    const modal = document.getElementById('membershipModal');
    const balanceValue = document.getElementById('membershipBalanceValue');
    
    if (modal && balanceValue) {
        balanceValue.textContent = formatCurrency(saldo);
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
    }
}

// Fechar modal de membership
function closeMembershipModal() {
    const modal = document.getElementById('membershipModal');
    modal.style.display = 'none';
    document.body.style.overflow = '';
    
    // Salvar que fechou hoje (para não mostrar de novo hoje)
    const today = new Date().toDateString();
    localStorage.setItem('membershipModalClosedDate', today);
}

// Upgrade para Premium
function upgradeToPremium() {
    // Redirecionar para página de pagamento
    window.location.href = CONFIG.payment.membership;
}
