// Elementos DOM
const profilePhoto = document.getElementById('profilePhoto');
const photoInput = document.getElementById('photoInput');
const userName = document.getElementById('userName');
const userWhatsApp = document.getElementById('userWhatsApp');
const userBalance = document.getElementById('userBalance');
const inputNome = document.getElementById('inputNome');
const inputIdade = document.getElementById('inputIdade');
const inputSexo = document.getElementById('inputSexo');
const inputWhatsApp = document.getElementById('inputWhatsApp');
const inputPixKey = document.getElementById('inputPixKey');
const btnSave = document.getElementById('btnSave');
const btnLogout = document.getElementById('btnLogout');

let currentUserData = null;
let onlineInterval;

// Atualizar status online
async function updateOnlineStatus() {
    try {
        await fetch('/mensagem/backend/update_online_status.php');
    } catch (error) {
        console.error('Erro ao atualizar status:', error);
    }
}

// Iniciar atualizações de status
function startOnlineUpdates() {
    updateOnlineStatus();
    onlineInterval = setInterval(updateOnlineStatus, 30000);
}

window.addEventListener('beforeunload', () => {
    if (onlineInterval) clearInterval(onlineInterval);
});

// Carregar dados do usuário
async function loadUserData() {
    try {
        const response = await fetch('/mensagem/backend/get_user_data.php');
        const data = await response.json();
        
        if (data.success) {
            currentUserData = data.user;
            displayUserData(data.user);
        } else {
            window.location.href = '/mensagem/autenticacao/index.html';
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao carregar dados', 'error');
    }
}

// Exibir dados do usuário
function displayUserData(user) {
    // Header
    userName.textContent = user.nome || 'Usuário';
    userWhatsApp.innerHTML = `<i class="fab fa-whatsapp"></i> ${user.whatsapp}`;
    
    // Foto de perfil
    if (user.foto_perfil) {
        profilePhoto.src = `/mensagem/${user.foto_perfil}`;
    }
    
    // Inputs
    inputNome.value = user.nome || '';
    inputIdade.value = user.idade || '';
    inputSexo.value = user.sexo || '';
    inputWhatsApp.value = user.whatsapp || '';
    
    // Saldo
    userBalance.textContent = formatCurrency(user.saldo || 0);
    
    // Chave PIX
    if (user.chave_pix) {
        inputPixKey.value = `${user.tipo_chave_pix?.toUpperCase() || 'PIX'}: ${user.chave_pix}`;
    } else {
        inputPixKey.value = 'Nenhuma chave cadastrada';
    }
}

// Formatar moeda
function formatCurrency(value) {
    return new Intl.NumberFormat('pt-BR', {
        style: 'currency',
        currency: 'BRL'
    }).format(value);
}

// Upload de foto
photoInput.addEventListener('change', async (e) => {
    const file = e.target.files[0];
    if (!file) return;
    
    // Validações
    const allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
    if (!allowedTypes.includes(file.type)) {
        showNotification('Apenas imagens JPG, PNG ou GIF', 'error');
        return;
    }
    
    if (file.size > 3 * 1024 * 1024) {
        showNotification('Imagem deve ter no máximo 3MB', 'error');
        return;
    }
    
    // Preview
    const reader = new FileReader();
    reader.onload = (e) => {
        profilePhoto.src = e.target.result;
    };
    reader.readAsDataURL(file);
    
    // Upload
    const formData = new FormData();
    formData.append('foto_perfil', file);
    
    try {
        const response = await fetch('/mensagem/backend/update_profile.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification('Foto atualizada com sucesso!', 'success');
        } else {
            showNotification(data.message || 'Erro ao atualizar foto', 'error');
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao enviar foto', 'error');
    }
});

// Salvar alterações
btnSave.addEventListener('click', async () => {
    const nome = inputNome.value.trim();
    const idade = inputIdade.value;
    const sexo = inputSexo.value;
    
    // Validações
    if (!nome) {
        showNotification('Digite seu nome', 'error');
        return;
    }
    
    if (!idade || idade < 16 || idade > 80) {
        showNotification('Idade deve estar entre 16 e 80 anos', 'error');
        return;
    }
    
    if (!sexo) {
        showNotification('Selecione seu sexo', 'error');
        return;
    }
    
    const formData = new FormData();
    formData.append('nome', nome);
    formData.append('idade', idade);
    formData.append('sexo', sexo);
    
    try {
        const response = await fetch('/mensagem/backend/update_profile.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification('Perfil atualizado com sucesso!', 'success');
            userName.textContent = nome;
        } else {
            showNotification(data.message || 'Erro ao atualizar perfil', 'error');
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao salvar alterações', 'error');
    }
});

// Logout
btnLogout.addEventListener('click', async () => {
    if (!confirm('Deseja realmente sair da sua conta?')) {
        return;
    }
    
    try {
        const response = await fetch('/mensagem/backend/logout.php');
        const data = await response.json();
        
        if (data.success) {
            window.location.href = '/mensagem/autenticacao/index.html';
        } else {
            showNotification('Erro ao sair', 'error');
        }
    } catch (error) {
        console.error('Erro:', error);
        // Forçar logout mesmo com erro
        window.location.href = '/mensagem/autenticacao/index.html';
    }
});

// Sistema de notificações
function showNotification(message, type = 'info') {
    const existing = document.querySelector('.notification');
    if (existing) existing.remove();
    
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
        <span>${message}</span>
    `;
    
    notification.style.cssText = `
        position: fixed;
        top: 90px;
        right: 24px;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
        color: white;
        padding: 16px 24px;
        border-radius: 12px;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        display: flex;
        align-items: center;
        gap: 12px;
        font-size: 14px;
        font-weight: 600;
        z-index: 9999;
        animation: slideInRight 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from { transform: translateX(400px); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOutRight {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(400px); opacity: 0; }
    }
`;
document.head.appendChild(style);

// Inicializar
loadUserData();
startOnlineUpdates();
