// Elementos DOM
const loginScreen = document.getElementById('loginScreen');
const adminPanel = document.getElementById('adminPanel');
const loginForm = document.getElementById('loginForm');
const navItems = document.querySelectorAll('.nav-item:not(.logout-btn)');
const logoutBtn = document.querySelector('.logout-btn');
const usersSection = document.getElementById('usersSection');
const conversationsSection = document.getElementById('conversationsSection');
const sectionTitle = document.getElementById('sectionTitle');

// Variáveis globais
let currentSection = 'users';
let userToDelete = null;

// Função auxiliar para escapar HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Verificar autenticação ao carregar
window.addEventListener('DOMContentLoaded', () => {
    checkAuth();
});

// Verificar se admin está autenticado
function checkAuth() {
    const isAdminAuthenticated = sessionStorage.getItem('adminAuth');
    if (isAdminAuthenticated === 'true') {
        showAdminPanel();
    }
}

// Login
loginForm.addEventListener('submit', async (e) => {
    e.preventDefault();
    
    const whatsapp = document.getElementById('adminWhatsApp').value;
    const password = document.getElementById('adminPassword').value;
    
    try {
        const formData = new FormData();
        formData.append('whatsapp', whatsapp);
        formData.append('password', password);
        
        const response = await fetch('/mensagem/backend/admin_login.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            sessionStorage.setItem('adminAuth', 'true');
            if (data.admin_name) {
                sessionStorage.setItem('adminName', data.admin_name);
            }
            showAdminPanel();
        } else {
            showNotification(data.message || 'Credenciais inválidas', 'error');
        }
    } catch (error) {
        console.error('Erro no login:', error);
        showNotification('Erro ao fazer login', 'error');
    }
});

// Mostrar painel admin
function showAdminPanel() {
    loginScreen.style.display = 'none';
    adminPanel.style.display = 'flex';
    loadUsers();
    loadStats();
}

// Logout
logoutBtn.addEventListener('click', (e) => {
    e.preventDefault();
    sessionStorage.removeItem('adminAuth');
    adminPanel.style.display = 'none';
    loginScreen.style.display = 'flex';
    loginForm.reset();
});

// Navegação entre seções
navItems.forEach(item => {
    item.addEventListener('click', (e) => {
        e.preventDefault();
        
        // Remover active de todos
        navItems.forEach(nav => nav.classList.remove('active'));
        
        // Adicionar active no clicado
        item.classList.add('active');
        
        // Mudar seção
        const section = item.dataset.section;
        currentSection = section;
        
        if (section === 'users') {
            sectionTitle.textContent = 'Gerenciar Usuários';
            usersSection.style.display = 'block';
            conversationsSection.style.display = 'none';
            loadUsers();
        } else if (section === 'conversations') {
            sectionTitle.textContent = 'Visualizar Conversas';
            usersSection.style.display = 'none';
            conversationsSection.style.display = 'block';
            loadConversations();
        }
    });
});

// Carregar estatísticas
async function loadStats() {
    try {
        const response = await fetch('/mensagem/backend/admin_stats.php');
        const data = await response.json();
        
        if (data.success) {
            document.getElementById('totalUsers').textContent = data.total_users;
            document.getElementById('totalConversations').textContent = data.total_conversations;
        }
    } catch (error) {
        console.error('Erro ao carregar stats:', error);
    }
}

// Carregar usuários
async function loadUsers() {
    const tbody = document.getElementById('usersTableBody');
    tbody.innerHTML = '<tr><td colspan="8" class="loading"><i class="fas fa-spinner fa-spin"></i> Carregando...</td></tr>';
    
    try {
        const response = await fetch('/mensagem/backend/admin_get_users.php');
        const data = await response.json();
        
        if (data.success && data.users.length > 0) {
            tbody.innerHTML = '';
            
            data.users.forEach(user => {
                const tr = document.createElement('tr');
                
                const isOnline = user.segundos_offline <= 30;
                const photoSrc = user.foto_perfil ? `/mensagem/${user.foto_perfil}` : '/mensagem/assets/logo.png';
                
                // Formatar sexo
                let sexoLabel = '-';
                if (user.sexo === 'masculino' || user.sexo === 'M') {
                    sexoLabel = '<i class="fas fa-mars" style="color: #3b82f6;"></i> Masculino';
                } else if (user.sexo === 'feminino' || user.sexo === 'F') {
                    sexoLabel = '<i class="fas fa-venus" style="color: #ec4899;"></i> Feminino';
                } else if (user.sexo === 'outro') {
                    sexoLabel = '<i class="fas fa-genderless" style="color: #8b5cf6;"></i> Outro';
                }
                
                // Formatar saldo
                const saldo = parseFloat(user.saldo || 0);
                const saldoFormatado = `R$ ${saldo.toFixed(2).replace('.', ',')}`;
                
                tr.innerHTML = `
                    <td><img src="${photoSrc}" alt="${user.nome}" class="user-photo" onerror="this.src='/mensagem/assets/logo.png'; this.onerror=null;"></td>
                    <td><strong>${user.nome}</strong></td>
                    <td>${user.whatsapp || 'Não informado'}</td>
                    <td>${user.idade || '-'}</td>
                    <td>${sexoLabel}</td>
                    <td>
                        <strong style="color: #059669;">${saldoFormatado}</strong>
                        <button class="btn-edit-saldo-inline" data-user-id="${user.id}" data-user-name="${user.nome}" data-user-saldo="${saldo}" title="Editar Saldo">
                            <i class="fas fa-edit"></i>
                        </button>
                    </td>
                    <td>
                        <span class="status-badge ${isOnline ? 'online' : 'offline'}">
                            <i class="fas fa-circle"></i>
                            ${isOnline ? 'Online' : 'Offline'}
                        </span>
                    </td>
                    <td>
                        <button class="btn-edit-photo" data-user-id="${user.id}" data-user-name="${user.nome}" data-photo-src="${photoSrc}" title="Alterar Foto">
                            <i class="fas fa-camera"></i>
                        </button>
                        <button class="btn-edit-profile" data-user-id="${user.id}" data-user-name="${user.nome}" data-user-whatsapp="${user.whatsapp || ''}" data-user-idade="${user.idade || ''}" data-user-sexo="${user.sexo || ''}" data-user-saldo="${user.saldo || 0}" title="Editar Perfil">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn-delete-user" onclick="openDeleteModal(${user.id}, '${user.nome}', '${user.foto_perfil || ''}')">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                `;
                
                tbody.appendChild(tr);
            });
        } else {
            tbody.innerHTML = '<tr><td colspan="8" class="loading">Nenhum usuário encontrado</td></tr>';
        }
    } catch (error) {
        console.error('Erro ao carregar usuários:', error);
        tbody.innerHTML = '<tr><td colspan="8" class="loading" style="color: #ef4444;">Erro ao carregar usuários</td></tr>';
    }
}

// Buscar usuários
document.getElementById('searchUsers').addEventListener('input', async (e) => {
    const search = e.target.value;
    
    if (search.length < 2) {
        loadUsers();
        return;
    }
    
    const tbody = document.getElementById('usersTableBody');
    
    try {
        const response = await fetch(`/mensagem/backend/admin_get_users.php?search=${encodeURIComponent(search)}`);
        const data = await response.json();
        
        if (data.success && data.users.length > 0) {
            tbody.innerHTML = '';
            
            data.users.forEach(user => {
                const tr = document.createElement('tr');
                
                const isOnline = user.segundos_offline <= 30;
                const photoSrc = user.foto_perfil ? `/mensagem/${user.foto_perfil}` : '/mensagem/assets/logo.png';
                
                // Formatar sexo
                let sexoLabel = '-';
                if (user.sexo === 'masculino' || user.sexo === 'M') {
                    sexoLabel = '<i class="fas fa-mars" style="color: #3b82f6;"></i> Masculino';
                } else if (user.sexo === 'feminino' || user.sexo === 'F') {
                    sexoLabel = '<i class="fas fa-venus" style="color: #ec4899;"></i> Feminino';
                } else if (user.sexo === 'outro') {
                    sexoLabel = '<i class="fas fa-genderless" style="color: #8b5cf6;"></i> Outro';
                }
                
                // Formatar saldo
                const saldo = parseFloat(user.saldo || 0);
                const saldoFormatado = `R$ ${saldo.toFixed(2).replace('.', ',')}`;
                
                tr.innerHTML = `
                    <td><img src="${photoSrc}" alt="${user.nome}" class="user-photo" onerror="this.src='/mensagem/assets/logo.png'; this.onerror=null;"></td>
                    <td><strong>${user.nome}</strong></td>
                    <td>${user.whatsapp || 'Não informado'}</td>
                    <td>${user.idade || '-'}</td>
                    <td>${sexoLabel}</td>
                    <td>
                        <strong style="color: #059669;">${saldoFormatado}</strong>
                        <button class="btn-edit-saldo-inline" data-user-id="${user.id}" data-user-name="${user.nome}" data-user-saldo="${saldo}" title="Editar Saldo">
                            <i class="fas fa-edit"></i>
                        </button>
                    </td>
                    <td>
                        <span class="status-badge ${isOnline ? 'online' : 'offline'}">
                            <i class="fas fa-circle"></i>
                            ${isOnline ? 'Online' : 'Offline'}
                        </span>
                    </td>
                    <td>
                        <button class="btn-edit-photo" data-user-id="${user.id}" data-user-name="${user.nome}" data-photo-src="${photoSrc}" title="Alterar Foto">
                            <i class="fas fa-camera"></i>
                        </button>
                        <button class="btn-edit-profile" data-user-id="${user.id}" data-user-name="${user.nome}" data-user-whatsapp="${user.whatsapp || ''}" data-user-idade="${user.idade || ''}" data-user-sexo="${user.sexo || ''}" data-user-saldo="${user.saldo || 0}" title="Editar Perfil">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn-delete-user" onclick="openDeleteModal(${user.id}, '${user.nome}', '${user.foto_perfil || ''}')">
                            <i class="fas fa-trash"></i>
                            Excluir
                        </button>
                    </td>
                `;
                
                tbody.appendChild(tr);
            });
        } else {
            tbody.innerHTML = '<tr><td colspan="8" class="loading">Nenhum usuário encontrado</td></tr>';
        }
    } catch (error) {
        console.error('Erro ao buscar usuários:', error);
    }
});

// Abrir modal de exclusão
function openDeleteModal(userId, userName, userPhoto) {
    userToDelete = { id: userId, name: userName, photo: userPhoto };
    document.getElementById('deleteModal').style.display = 'flex';
}

// Fechar modal de exclusão
function closeDeleteModal() {
    userToDelete = null;
    document.getElementById('deleteModal').style.display = 'none';
}

// Confirmar exclusão
async function confirmDelete() {
    if (!userToDelete) return;
    
    try {
        const formData = new FormData();
        formData.append('user_id', userToDelete.id);
        formData.append('photo_path', userToDelete.photo);
        
        const response = await fetch('/mensagem/backend/admin_delete_user.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification(`Usuário ${userToDelete.name} excluído com sucesso`, 'success');
            closeDeleteModal();
            loadUsers();
            loadStats();
        } else {
            showNotification(data.message || 'Erro ao excluir usuário', 'error');
        }
    } catch (error) {
        console.error('Erro ao excluir usuário:', error);
        showNotification('Erro ao excluir usuário', 'error');
    }
}

// Carregar conversas
async function loadConversations() {
    const list = document.getElementById('conversationsList');
    list.innerHTML = '<div class="loading"><i class="fas fa-spinner fa-spin"></i> Carregando...</div>';
    
    try {
        const response = await fetch('/mensagem/backend/admin_get_conversations.php');
        const data = await response.json();
        
        if (data.success && data.conversations.length > 0) {
            list.innerHTML = '';
            
            data.conversations.forEach(conv => {
                const div = document.createElement('div');
                div.className = 'conversation-item';
                div.onclick = () => loadMessages(conv.id, conv.user1_nome, conv.user2_nome);
                
                div.innerHTML = `
                    <i class="fas fa-comments" style="color: #667eea; font-size: 20px;"></i>
                    <div class="conversation-users">
                        <strong>${conv.user1_nome} & ${conv.user2_nome}</strong>
                        <span>${conv.total_mensagens} mensagens</span>
                    </div>
                `;
                
                list.appendChild(div);
            });
        } else {
            list.innerHTML = '<div class="loading">Nenhuma conversa encontrada</div>';
        }
    } catch (error) {
        console.error('Erro ao carregar conversas:', error);
        list.innerHTML = '<div class="loading" style="color: #ef4444;">Erro ao carregar conversas</div>';
    }
}

// Carregar mensagens de uma conversa
async function loadMessages(conversaId, user1, user2) {
    // Marcar conversa como ativa
    document.querySelectorAll('.conversation-item').forEach(item => {
        item.classList.remove('active');
    });
    event.target.closest('.conversation-item').classList.add('active');
    
    const viewer = document.getElementById('messagesViewer');
    viewer.innerHTML = '<div class="loading"><i class="fas fa-spinner fa-spin"></i> Carregando mensagens...</div>';
    
    try {
        const response = await fetch(`/mensagem/backend/admin_get_messages.php?conversa_id=${conversaId}`);
        const data = await response.json();
        
        if (data.success && data.messages.length > 0) {
            viewer.innerHTML = '';
            
            data.messages.forEach(msg => {
                const div = document.createElement('div');
                div.className = msg.is_sent ? 'message-item sent' : 'message-item received';
                
                const date = new Date(msg.created_at);
                const timeStr = date.toLocaleTimeString('pt-BR', {
                    hour: '2-digit',
                    minute: '2-digit'
                });
                
                let messageContent = '';
                
                // Renderizar mídia se existir
                if (msg.tipo_midia && msg.caminho_midia) {
                    if (msg.tipo_midia === 'image') {
                        messageContent = `
                            <div class="message-media">
                                <img src="/mensagem/${msg.caminho_midia}" alt="Imagem" style="max-width: 280px; max-height: 400px; border-radius: 8px; cursor: pointer;" onclick="window.open('/mensagem/${msg.caminho_midia}', '_blank')">
                            </div>
                        `;
                    } else if (msg.tipo_midia === 'video') {
                        messageContent = `
                            <div class="message-media">
                                <video src="/mensagem/${msg.caminho_midia}" controls style="max-width: 280px; max-height: 400px; border-radius: 8px;"></video>
                            </div>
                        `;
                    }
                }
                
                // Adicionar texto da mensagem se existir
                const messageText = msg.mensagem ? `<div class="message-text">${escapeHtml(msg.mensagem)}</div>` : '';
                
                div.innerHTML = `
                    <div class="message-bubble">
                        <div class="message-sender">${msg.sender_nome}</div>
                        ${messageContent}
                        ${messageText}
                        <div class="message-footer">
                            <span class="message-time">${timeStr}</span>
                        </div>
                    </div>
                `;
                
                viewer.appendChild(div);
            });
            
            // Scroll para o final
            viewer.scrollTop = viewer.scrollHeight;
        } else {
            viewer.innerHTML = '<div class="empty-state"><i class="fas fa-comment-slash"></i><p>Nenhuma mensagem encontrada</p></div>';
        }
    } catch (error) {
        console.error('Erro ao carregar mensagens:', error);
        viewer.innerHTML = '<div class="empty-state" style="color: #ef4444;"><i class="fas fa-exclamation-triangle"></i><p>Erro ao carregar mensagens</p></div>';
    }
}

// Notificação
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 30px;
        right: 30px;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
        color: white;
        padding: 18px 24px;
        border-radius: 12px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        z-index: 99999;
        font-weight: 600;
        animation: slideInRight 0.4s ease;
    `;
    
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
        ${message}
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.4s ease';
        setTimeout(() => notification.remove(), 400);
    }, 3000);
}

// Animações CSS
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from {
            opacity: 0;
            transform: translateX(100px);
        }
        to {
            opacity: 1;
            transform: translateX(0);
        }
    }
    
    @keyframes slideOutRight {
        from {
            opacity: 1;
            transform: translateX(0);
        }
        to {
            opacity: 0;
            transform: translateX(100px);
        }
    }
`;
document.head.appendChild(style);

// ===== LIMPAR TODAS AS MENSAGENS =====
const btnClearMessages = document.getElementById('btnClearMessages');
const clearMessagesModal = document.getElementById('clearMessagesModal');
const confirmClearInput = document.getElementById('confirmClearInput');

if (btnClearMessages) {
    btnClearMessages.addEventListener('click', () => {
        clearMessagesModal.style.display = 'flex';
        confirmClearInput.value = '';
    });
}

function closeClearMessagesModal() {
    clearMessagesModal.style.display = 'none';
    confirmClearInput.value = '';
}

async function confirmClearMessages() {
    const inputValue = confirmClearInput.value.trim();
    
    if (inputValue !== 'LIMPAR TUDO') {
        showNotification('Digite exatamente: LIMPAR TUDO', 'error');
        return;
    }
    
    if (!confirm('⚠️ ÚLTIMA CONFIRMAÇÃO: Tem ABSOLUTA certeza? Esta ação é IRREVERSÍVEL!')) {
        return;
    }
    
    try {
        showNotification('Limpando mensagens... Isso pode levar alguns minutos.', 'info');
        
        const response = await fetch('/mensagem/backend/admin_clear_messages.php', {
            method: 'POST'
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification(`✅ ${data.message}`, 'success');
            closeClearMessagesModal();
            
            // Recarregar estatísticas
            if (currentSection === 'conversations') {
                loadConversations();
            }
        } else {
            showNotification(data.message || 'Erro ao limpar mensagens', 'error');
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao processar requisição', 'error');
    }
}

// Fechar modal ao clicar fora
clearMessagesModal?.addEventListener('click', (e) => {
    if (e.target === clearMessagesModal) {
        closeClearMessagesModal();
    }
});

// ===== EDIT PHOTO MODAL =====
let userToEditPhoto = null;
let selectedPhotoFile = null;

function openEditPhotoModal(userId, userName, currentPhoto) {
    try {
        console.log('Abrindo modal para usuário:', userId, userName);
        
        userToEditPhoto = { id: userId, name: userName, currentPhoto: currentPhoto };
        
        const modal = document.getElementById('editPhotoModal');
        if (!modal) {
            console.error('Modal editPhotoModal não encontrado!');
            return;
        }
        
        const modalUserName = document.getElementById('editPhotoUserName');
        const currentPhotoPreview = document.getElementById('currentPhotoPreview');
        const newPhotoPreview = document.getElementById('newPhotoPreview');
        const photoPlaceholder = document.getElementById('photoPlaceholder');
        
        if (modalUserName) modalUserName.textContent = userName;
        if (currentPhotoPreview) currentPhotoPreview.src = currentPhoto;
        if (newPhotoPreview) {
            newPhotoPreview.style.display = 'none';
            newPhotoPreview.src = '';
        }
        if (photoPlaceholder) {
            photoPlaceholder.style.display = 'flex';
        }
        selectedPhotoFile = null;
        
        modal.classList.add('show');
        document.body.style.overflow = 'hidden';
        
        console.log('Modal aberto com sucesso!');
    } catch (error) {
        console.error('Erro ao abrir modal:', error);
    }
}

function closeEditPhotoModal() {
    const modal = document.getElementById('editPhotoModal');
    modal.classList.remove('show');
    document.body.style.overflow = '';
    userToEditPhoto = null;
    selectedPhotoFile = null;
}

function selectPhotoFile() {
    document.getElementById('photoFileInput').click();
}

function handlePhotoFileSelect(event) {
    const file = event.target.files[0];
    if (!file) return;
    
    // Validar tipo de arquivo
    if (!file.type.startsWith('image/')) {
        showNotification('Por favor, selecione uma imagem válida', 'error');
        return;
    }
    
    // Validar tamanho (max 5MB)
    if (file.size > 5 * 1024 * 1024) {
        showNotification('A imagem deve ter no máximo 5MB', 'error');
        return;
    }
    
    selectedPhotoFile = file;
    
    // Mostrar preview
    const reader = new FileReader();
    reader.onload = function(e) {
        const newPhotoPreview = document.getElementById('newPhotoPreview');
        const photoPlaceholder = document.getElementById('photoPlaceholder');
        
        newPhotoPreview.src = e.target.result;
        newPhotoPreview.style.display = 'block';
        if (photoPlaceholder) {
            photoPlaceholder.style.display = 'none';
        }
    };
    reader.readAsDataURL(file);
}

async function confirmUpdatePhoto() {
    if (!userToEditPhoto || !selectedPhotoFile) {
        showNotification('Por favor, selecione uma foto', 'error');
        return;
    }
    
    const formData = new FormData();
    formData.append('user_id', userToEditPhoto.id);
    formData.append('foto_perfil', selectedPhotoFile);
    
    try {
        const response = await fetch('/mensagem/backend/admin_update_photo.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification(`Foto de ${userToEditPhoto.name} atualizada com sucesso!`, 'success');
            closeEditPhotoModal();
            loadUsers(); // Recarregar lista
            loadAllUsers(); // Recarregar lista de todos usuários
        } else {
            showNotification(data.message || 'Erro ao atualizar foto', 'error');
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao processar requisição', 'error');
    }
}

// Event listener para botões de editar foto (usando delegação de eventos)
document.addEventListener('click', function(e) {
    const btn = e.target.closest('.btn-edit-photo');
    if (btn) {
        const userId = btn.getAttribute('data-user-id');
        const userName = btn.getAttribute('data-user-name');
        const photoSrc = btn.getAttribute('data-photo-src');
        
        console.log('Botão clicado:', { userId, userName, photoSrc });
        openEditPhotoModal(userId, userName, photoSrc);
    }
});


// ===== EDIT PROFILE MODAL =====
let userToEditProfile = null;

// Event listener para botões de editar perfil
document.addEventListener('click', function(e) {
    const btn = e.target.closest('.btn-edit-profile');
    if (btn) {
        const userId = btn.getAttribute('data-user-id');
        const userName = btn.getAttribute('data-user-name');
        const userWhatsApp = btn.getAttribute('data-user-whatsapp');
        const userIdade = btn.getAttribute('data-user-idade');
        const userSexo = btn.getAttribute('data-user-sexo');
        const userSaldo = btn.getAttribute('data-user-saldo');
        
        openEditProfileModal(userId, userName, userWhatsApp, userIdade, userSexo, userSaldo);
    }
});

function openEditProfileModal(userId, userName, whatsapp, idade, sexo, saldo) {
    try {
        console.log('Abrindo modal de edição para:', userId, userName);
        
        userToEditProfile = { id: userId, name: userName };
        
        const modal = document.getElementById('editProfileModal');
        if (!modal) {
            console.error('Modal editProfileModal não encontrado!');
            return;
        }
        
        // Preencher campos
        document.getElementById('editProfileUserName').textContent = userName;
        document.getElementById('editNome').value = userName || '';
        document.getElementById('editWhatsApp').value = whatsapp || '';
        document.getElementById('editIdade').value = idade || '';
        document.getElementById('editSexo').value = sexo || '';
        document.getElementById('editSaldo').value = saldo || '0';
        document.getElementById('editSenha').value = '';
        
        modal.classList.add('show');
        document.body.style.overflow = 'hidden';
        
        console.log('Modal de edição aberto com sucesso!');
    } catch (error) {
        console.error('Erro ao abrir modal:', error);
    }
}

function closeEditProfileModal() {
    const modal = document.getElementById('editProfileModal');
    modal.classList.remove('show');
    document.body.style.overflow = '';
    userToEditProfile = null;
}

async function confirmUpdateProfile() {
    if (!userToEditProfile) {
        showNotification('Erro: usuário não selecionado', 'error');
        return;
    }
    
    const nome = document.getElementById('editNome').value.trim();
    const whatsapp = document.getElementById('editWhatsApp').value.trim();
    const idade = document.getElementById('editIdade').value.trim();
    const sexo = document.getElementById('editSexo').value;
    const saldo = document.getElementById('editSaldo').value.trim();
    const senha = document.getElementById('editSenha').value;
    
    // Validações
    if (!nome) {
        showNotification('Nome é obrigatório', 'error');
        return;
    }
    
    if (idade && (idade < 18 || idade > 120)) {
        showNotification('Idade deve estar entre 18 e 120 anos', 'error');
        return;
    }
    
    if (saldo && parseFloat(saldo) < 0) {
        showNotification('Saldo não pode ser negativo', 'error');
        return;
    }
    
    const formData = new FormData();
    formData.append('user_id', userToEditProfile.id);
    formData.append('nome', nome);
    if (whatsapp) formData.append('whatsapp', whatsapp);
    if (idade) formData.append('idade', idade);
    if (sexo) formData.append('sexo', sexo);
    if (saldo) formData.append('saldo', saldo);
    if (senha) formData.append('senha', senha);
    
    try {
        const response = await fetch('/mensagem/backend/admin_update_user.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification(`Perfil de ${userToEditProfile.name} atualizado com sucesso!`, 'success');
            closeEditProfileModal();
            loadUsers();
        } else {
            showNotification(data.message || 'Erro ao atualizar perfil', 'error');
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao processar requisição', 'error');
    }
}

// ===== MODAL DE EDIÇÃO DE SALDO =====
let userToEditSaldo = null;

// Event listener para botão de editar saldo
document.addEventListener('click', (e) => {
    const btn = e.target.closest('.btn-edit-saldo-inline');
    if (btn) {
        const userId = btn.getAttribute('data-user-id');
        const userName = btn.getAttribute('data-user-name');
        const userSaldo = btn.getAttribute('data-user-saldo');
        
        openEditSaldoModal(userId, userName, userSaldo);
    }
});

function openEditSaldoModal(userId, userName, saldo) {
    userToEditSaldo = { id: userId, name: userName };
    
    const modal = document.getElementById('editSaldoModal');
    const userNameElement = document.getElementById('editSaldoUserName');
    const currentSaldoElement = document.getElementById('currentSaldoValue');
    const newSaldoInput = document.getElementById('newSaldoInput');
    
    // Preencher dados
    userNameElement.textContent = userName;
    
    const saldoNumerico = parseFloat(saldo) || 0;
    currentSaldoElement.textContent = `R$ ${saldoNumerico.toFixed(2).replace('.', ',')}`;
    
    newSaldoInput.value = saldoNumerico.toFixed(2);
    
    // Mostrar modal
    modal.classList.add('show');
    document.body.style.overflow = 'hidden';
    
    // Focar no input
    setTimeout(() => {
        newSaldoInput.select();
    }, 100);
}

function closeEditSaldoModal() {
    const modal = document.getElementById('editSaldoModal');
    modal.classList.remove('show');
    document.body.style.overflow = '';
    userToEditSaldo = null;
}

function setQuickSaldo(valor) {
    const input = document.getElementById('newSaldoInput');
    input.value = valor.toFixed(2);
    input.focus();
}

async function confirmUpdateSaldo() {
    if (!userToEditSaldo) {
        showNotification('Erro: usuário não selecionado', 'error');
        return;
    }
    
    const newSaldo = document.getElementById('newSaldoInput').value.trim();
    
    // Validações
    if (!newSaldo) {
        showNotification('Digite o novo saldo', 'error');
        return;
    }
    
    const saldoNumerico = parseFloat(newSaldo);
    
    if (isNaN(saldoNumerico) || saldoNumerico < 0) {
        showNotification('Saldo inválido ou negativo', 'error');
        return;
    }
    
    const formData = new FormData();
    formData.append('user_id', userToEditSaldo.id);
    formData.append('nome', userToEditSaldo.name); // Nome obrigatório no backend
    formData.append('saldo', saldoNumerico);
    
    try {
        const response = await fetch('/mensagem/backend/admin_update_user.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification(`Saldo de ${userToEditSaldo.name} atualizado para R$ ${saldoNumerico.toFixed(2).replace('.', ',')}!`, 'success');
            closeEditSaldoModal();
            loadUsers();
        } else {
            showNotification(data.message || 'Erro ao atualizar saldo', 'error');
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao processar requisição', 'error');
    }
}
