/**
 * Kwai Ads Tracking Helper
 * Script auxiliar para integrar tracking do Kwai nas páginas do site
 */

class KwaiTracker {
    constructor() {
        this.baseUrl = '../backend/';
        this.ready = false;
        this.initPromise = this.init();
    }

    /**
     * Inicializa o tracker
     */
    async init() {
        // Captura click_id se presente na URL
        const urlParams = new URLSearchParams(window.location.search);
        const clickId = urlParams.get('click_id');
        
        if (clickId) {
            await this.captureClickId(clickId);
        }
        
        this.ready = true;
        return true;
    }
    
    /**
     * Aguarda o tracker estar pronto
     */
    async waitReady() {
        await this.initPromise;
        return this.ready;
    }

    /**
     * Captura e armazena o click_id
     */
    async captureClickId(clickId) {
        try {
            const response = await fetch(`${this.baseUrl}kwai_capture_clickid.php?click_id=${encodeURIComponent(clickId)}`);
            const data = await response.json();
            
            if (data.success) {
                console.log('Kwai click_id capturado:', clickId);
                return true;
            } else {
                console.error('Erro ao capturar click_id:', data.message);
                return false;
            }
        } catch (error) {
            console.error('Erro ao capturar click_id:', error);
            return false;
        }
    }

    /**
     * Envia evento de registro
     */
    async trackRegistration() {
        return await this.sendEvent('registration');
    }

    /**
     * Envia evento de compra
     */
    async trackPurchase(value, orderId = null, products = []) {
        return await this.sendEvent('purchase', {
            value: value,
            order_id: orderId,
            products: products
        });
    }

    /**
     * Envia evento de início de checkout
     */
    async trackInitiatedCheckout(value = null) {
        return await this.sendEvent('initiated_checkout', {
            value: value
        });
    }

    /**
     * Envia evento de adição de pagamento
     */
    async trackAddPaymentInfo() {
        return await this.sendEvent('add_payment_info');
    }

    /**
     * Envia evento de visualização de página
     */
    async trackPageView(pageName) {
        return await this.sendEvent('page_view', {
            page_name: pageName
        });
    }

    /**
     * Envia evento de primeiro depósito
     */
    async trackFirstDeposit(value) {
        return await this.sendEvent('first_deposit', {
            value: value
        });
    }

    /**
     * Envia evento customizado
     */
    async trackCustomEvent(eventName, properties = {}, value = null, currency = null) {
        return await this.sendEvent('custom', {
            event_name: eventName,
            properties: properties,
            value: value,
            currency: currency
        });
    }

    /**
     * Função genérica para enviar eventos
     */
    async sendEvent(eventType, data = {}) {
        try {
            const payload = {
                event_type: eventType,
                ...data
            };

            const response = await fetch(`${this.baseUrl}kwai_send_event.php`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(payload)
            });

            const result = await response.json();

            if (result.success) {
                console.log(`Kwai evento enviado: ${eventType}`);
                return true;
            } else {
                console.error(`Erro ao enviar evento ${eventType}:`, result.message);
                return false;
            }
        } catch (error) {
            console.error(`Erro ao enviar evento ${eventType}:`, error);
            return false;
        }
    }
}

// Inicializa automaticamente quando o script é carregado
const kwaiTracker = new KwaiTracker();

// Exemplo de uso:
/*
// Após registro do usuário:
kwaiTracker.trackRegistration();

// Após compra:
kwaiTracker.trackPurchase(150.00, 'ORDER123', [
    { id: 'PROD1', quantity: 2, price: 50.00 },
    { id: 'PROD2', quantity: 1, price: 50.00 }
]);

// Ao iniciar checkout:
kwaiTracker.trackInitiatedCheckout(150.00);

// Ao visualizar página:
kwaiTracker.trackPageView('Homepage');

// Evento customizado:
kwaiTracker.trackCustomEvent('EVENT_CONTACT', {
    content_name: 'Contact Form',
    content_id: 'form_01'
});
*/
