<?php
/**
 * INSTALADOR SIMPLIFICADO DO SISTEMA
 * Versão sem streaming para melhor compatibilidade
 */

require_once __DIR__ . '/backend/config_global.php';

// Verificar se já foi instalado
$alreadyInstalled = false;
try {
    $conn = getConnection();
    $stmt = $conn->query("SHOW TABLES LIKE 'usuarios'");
    if ($stmt->rowCount() > 0) {
        $alreadyInstalled = true;
    }
} catch (Exception $e) {
    // Banco não existe ou não tem permissão
}

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalação do Sistema</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            max-width: 600px;
            width: 100%;
            padding: 40px;
        }
        
        h1 {
            color: #2d3748;
            margin-bottom: 10px;
            font-size: 28px;
        }
        
        .subtitle {
            color: #718096;
            margin-bottom: 30px;
            font-size: 16px;
        }
        
        .status {
            background: #f7fafc;
            border-left: 4px solid #667eea;
            padding: 20px;
            margin-bottom: 20px;
            border-radius: 8px;
        }
        
        .status h3 {
            color: #2d3748;
            margin-bottom: 10px;
            font-size: 18px;
        }
        
        .success {
            background: #f0fdf4;
            border-left-color: #10b981;
        }
        
        .error {
            background: #fef2f2;
            border-left-color: #ef4444;
        }
        
        .warning {
            background: #fffbeb;
            border-left-color: #f59e0b;
        }
        
        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 15px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s;
            width: 100%;
            margin-top: 20px;
        }
        
        .btn:hover:not(:disabled) {
            transform: translateY(-2px);
        }
        
        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        .log {
            background: #1a202c;
            color: #68d391;
            padding: 20px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
            max-height: 400px;
            overflow-y: auto;
            margin-top: 20px;
        }
        
        .log-line {
            margin-bottom: 8px;
        }
        
        .log-success { color: #68d391; }
        .log-error { color: #fc8181; }
        .log-warning { color: #f6ad55; }
        .log-info { color: #63b3ed; }
        
        .spinner {
            border: 3px solid #f3f3f3;
            border-top: 3px solid #667eea;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 20px auto;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 Instalação do Sistema</h1>
        <p class="subtitle">Configure automaticamente o banco de dados</p>
        
        <?php if ($alreadyInstalled): ?>
            <div class="status warning">
                <h3>⚠️ Sistema já instalado!</h3>
                <p>As tabelas já existem no banco de dados.</p>
                <p style="margin-top: 10px;">
                    <strong>Opções:</strong><br>
                    • Se deseja reinstalar, delete as tabelas manualmente no banco<br>
                    • Se está tudo OK, acesse o sistema normalmente
                </p>
            </div>
            <button class="btn" onclick="window.location.href='<?php echo getUrl('autenticacao/index.html'); ?>'">
                Acessar Sistema
            </button>
        <?php else: ?>
            <div id="statusBox" class="status">
                <h3>✓ Conexão verificada</h3>
                <p>Pronto para instalar as tabelas do banco de dados</p>
            </div>
            
            <button class="btn" id="installBtn" onclick="startInstallation()">
                Iniciar Instalação
            </button>
            
            <div id="logBox" style="display: none;"></div>
        <?php endif; ?>
    </div>
    
    <script>
        function log(message, type = 'info') {
            const logBox = document.getElementById('logBox');
            logBox.style.display = 'block';
            if (!logBox.classList.contains('log')) {
                logBox.classList.add('log');
            }
            const line = document.createElement('div');
            line.className = `log-line log-${type}`;
            const time = new Date().toLocaleTimeString();
            line.textContent = `[${time}] ${message}`;
            logBox.appendChild(line);
            logBox.scrollTop = logBox.scrollHeight;
        }
        
        async function startInstallation() {
            const btn = document.getElementById('installBtn');
            const statusBox = document.getElementById('statusBox');
            
            btn.disabled = true;
            btn.textContent = 'Instalando...';
            
            statusBox.innerHTML = '<div class="spinner"></div><p style="text-align: center;">Instalando banco de dados...</p>';
            
            log('Iniciando instalação...', 'info');
            
            try {
                const response = await fetch('install_execute.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ action: 'install' })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    log('✓ Instalação concluída com sucesso!', 'success');
                    statusBox.className = 'status success';
                    statusBox.innerHTML = '<h3>✓ Instalação Concluída!</h3><p>Todas as tabelas foram criadas com sucesso.</p>';
                    
                    btn.textContent = 'Acessar Sistema';
                    btn.disabled = false;
                    btn.onclick = function() {
                        window.location.href = data.redirect;
                    };
                    
                    // Redirecionar automaticamente em 3 segundos
                    setTimeout(() => {
                        window.location.href = data.redirect;
                    }, 3000);
                } else {
                    throw new Error(data.message || 'Erro desconhecido');
                }
            } catch (error) {
                log('✗ Erro: ' + error.message, 'error');
                statusBox.className = 'status error';
                statusBox.innerHTML = '<h3>✗ Erro na Instalação</h3><p>' + error.message + '</p>';
                btn.disabled = false;
                btn.textContent = 'Tentar Novamente';
            }
        }
    </script>
</body>
</html>
