<?php
/**
 * Executor da instalação (sem streaming)
 */

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/backend/config_global.php';

try {
    // Conectar ao banco com buffered queries
    $conn = getConnection();
    $conn->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);
    $conn->setAttribute(PDO::MYSQL_ATTR_USE_BUFFERED_QUERY, true);
    
    // Ler arquivo SQL
    $sqlFile = __DIR__ . '/database_setup.sql';
    
    if (!file_exists($sqlFile)) {
        throw new Exception('Arquivo database_setup.sql não encontrado');
    }
    
    $sql = file_get_contents($sqlFile);
    
    // Remover comentários -- (linha única)
    $sql = preg_replace('/--[^\r\n]*/', '', $sql);
    
    // Remover comentários /* ... */ (multi-linha)
    $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);
    
    // Dividir por ponto-e-vírgula
    $statements = explode(';', $sql);
    
    // Executar statements
    $executed = 0;
    $errors = [];
    
    foreach ($statements as $statement) {
        $statement = trim($statement);
        
        // Pular vazios
        if (empty($statement)) {
            continue;
        }
        
        // Pular SELECTs e USEs
        $firstWord = strtoupper(substr($statement, 0, 6));
        if ($firstWord === 'SELECT' || $firstWord === 'USE' || substr($firstWord, 0, 3) === 'USE') {
            continue;
        }
        
        try {
            $result = $conn->exec($statement);
            $executed++;
        } catch (PDOException $e) {
            // Ignorar erros de "tabela já existe"
            if (strpos($e->getMessage(), 'already exists') === false &&
                strpos($e->getMessage(), 'Duplicate') === false) {
                $errors[] = substr($statement, 0, 50) . '... : ' . $e->getMessage();
            }
        }
    }
    
    // Verificar quantas tabelas foram criadas
    $result = $conn->query("SHOW TABLES");
    $tables = $result->fetchAll(PDO::FETCH_COLUMN);
    $tableCount = count($tables);
    
    if ($tableCount === 0) {
        throw new Exception('Nenhuma tabela foi criada! Verifique as permissões do banco de dados. Erros: ' . implode('; ', $errors));
    }
    
    echo json_encode([
        'success' => true,
        'message' => "Instalação concluída! $tableCount tabelas criadas no banco de dados.",
        'executed' => $executed,
        'tables' => $tables,
        'errors' => $errors,
        'redirect' => getUrl('autenticacao/index.html')
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
