-- ================================================================
-- SCRIPT COMPLETO PARA CRIAR BANCO DE DADOS - PLATAFORMA DE CHAT
-- ================================================================
-- 
-- Execute este script em seu banco MySQL/MariaDB para criar
-- todas as tabelas e estruturas necessárias para o funcionamento
-- completo da plataforma de chat com sistema de ganhos.
--
-- Versão: 1.0
-- Data: 2025-11-06
-- ================================================================

-- Criar banco de dados (opcional - descomente se necessário)
-- CREATE DATABASE IF NOT EXISTS mensagem_chat CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
-- USE mensagem_chat;

-- ================================================================
-- 1. TABELA DE USUÁRIOS (Principal)
-- ================================================================

CREATE TABLE IF NOT EXISTS usuarios (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(255) NOT NULL,
    whatsapp VARCHAR(20) NOT NULL UNIQUE,
    senha VARCHAR(255) NOT NULL,
    
    -- Campos de perfil
    foto_perfil VARCHAR(255) NULL,
    idade INT NULL,
    sexo ENUM('masculino', 'feminino', 'outro') NULL,
    perfil_completo TINYINT(1) DEFAULT 0,
    
    -- Sistema financeiro
    saldo DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    is_member TINYINT(1) DEFAULT 0,
    
    -- Sistema PIX para saques
    chave_pix VARCHAR(255) NULL,
    tipo_chave_pix ENUM('cpf', 'cnpj', 'email', 'telefone', 'aleatorio') NULL,
    
    -- Sistema de controle
    bloqueado TINYINT(1) DEFAULT 0,
    avisos_palavras INT DEFAULT 0,
    
    -- Status online
    last_seen TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Timestamps
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Índices para performance
    INDEX idx_whatsapp (whatsapp),
    INDEX idx_last_seen (last_seen),
    INDEX idx_bloqueado (bloqueado),
    INDEX idx_is_member (is_member)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ================================================================
-- 2. TABELA DE CONVERSAS
-- ================================================================

CREATE TABLE IF NOT EXISTS conversas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user1_id INT NOT NULL,
    user2_id INT NOT NULL,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Chave única para evitar conversas duplicadas
    UNIQUE KEY unique_conversation (user1_id, user2_id),
    
    -- Chaves estrangeiras
    FOREIGN KEY (user1_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    FOREIGN KEY (user2_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Índices para performance
    INDEX idx_user1 (user1_id),
    INDEX idx_user2 (user2_id),
    INDEX idx_updated (updated_at)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ================================================================
-- 3. TABELA DE MENSAGENS
-- ================================================================

CREATE TABLE IF NOT EXISTS mensagens (
    id INT AUTO_INCREMENT PRIMARY KEY,
    conversa_id INT NOT NULL,
    remetente_id INT NOT NULL,
    destinatario_id INT NOT NULL,
    
    -- Conteúdo da mensagem
    mensagem TEXT NOT NULL,
    
    -- Sistema de mídia
    tipo_midia ENUM('texto', 'imagem', 'audio', 'video') DEFAULT 'texto',
    caminho_midia VARCHAR(500) NULL,
    
    -- Sistema de presentes
    gift_amount DECIMAL(10,2) DEFAULT NULL,
    
    -- Sistema de resposta
    reply_to_id INT NULL,
    
    -- Status da mensagem
    lida TINYINT(1) DEFAULT 0,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Chaves estrangeiras
    FOREIGN KEY (conversa_id) REFERENCES conversas(id) ON DELETE CASCADE,
    FOREIGN KEY (remetente_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    FOREIGN KEY (destinatario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    FOREIGN KEY (reply_to_id) REFERENCES mensagens(id) ON DELETE SET NULL,
    
    -- Índices para performance
    INDEX idx_conversa (conversa_id),
    INDEX idx_remetente (remetente_id),
    INDEX idx_destinatario (destinatario_id),
    INDEX idx_lida (lida),
    INDEX idx_created (created_at),
    INDEX idx_gift (gift_amount)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ================================================================
-- 4. TABELA DE HISTÓRICO DE SALDO
-- ================================================================

CREATE TABLE IF NOT EXISTS saldo_historico (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    
    -- Tipo de transação
    tipo ENUM('gift_sent', 'gift_received', 'message_earn', 'withdrawal', 'payment', 'admin_adjustment') NOT NULL,
    
    -- Valor da transação (positivo = ganho, negativo = gasto)
    valor DECIMAL(10,2) NOT NULL,
    
    -- Descrição da transação
    descricao VARCHAR(255) NULL,
    
    -- ID relacionado (mensagem, saque, etc)
    relacionado_id INT DEFAULT NULL,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Chave estrangeira
    FOREIGN KEY (user_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Índices para performance
    INDEX idx_user_id (user_id),
    INDEX idx_tipo (tipo),
    INDEX idx_created_at (created_at),
    INDEX idx_relacionado (relacionado_id)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ================================================================
-- 5. TABELA DE SAQUES
-- ================================================================

CREATE TABLE IF NOT EXISTS saques (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    
    -- Dados do saque
    valor DECIMAL(10,2) NOT NULL,
    chave_pix VARCHAR(255) NOT NULL,
    tipo_chave_pix ENUM('cpf', 'cnpj', 'email', 'telefone', 'aleatorio') NOT NULL,
    
    -- Status do saque
    status ENUM('pendente', 'processando', 'aprovado', 'rejeitado') DEFAULT 'pendente',
    
    -- Observações
    observacoes TEXT NULL,
    
    -- Controle de datas
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    processed_at TIMESTAMP NULL,
    
    -- Chave estrangeira
    FOREIGN KEY (user_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Índices para performance
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ================================================================
-- 6. TABELA DE BLOQUEIOS
-- ================================================================

CREATE TABLE IF NOT EXISTS bloqueios (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    blocked_user_id INT NOT NULL,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Chaves estrangeiras
    FOREIGN KEY (user_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    FOREIGN KEY (blocked_user_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    
    -- Evitar bloqueios duplicados
    UNIQUE KEY unique_block (user_id, blocked_user_id),
    
    -- Índices para performance
    INDEX idx_user_id (user_id),
    INDEX idx_blocked_user_id (blocked_user_id)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ================================================================
-- 7. TABELA DE PALAVRAS PROIBIDAS
-- ================================================================

CREATE TABLE IF NOT EXISTS palavras_proibidas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    palavra VARCHAR(100) NOT NULL,
    ativo TINYINT(1) DEFAULT 1,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Evitar palavras duplicadas
    UNIQUE KEY unique_palavra (palavra),
    
    -- Índices para performance
    INDEX idx_ativo (ativo)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ================================================================
-- 8. TABELA DE TRACKING KWAI (opcional)
-- ================================================================

CREATE TABLE IF NOT EXISTS kwai_tracking (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NULL,
    click_id VARCHAR(255) NULL,
    conversion_id VARCHAR(255) NULL,
    
    -- Dados do evento
    event_type ENUM('registration', 'purchase', 'page_view') NOT NULL,
    event_value DECIMAL(10,2) NULL,
    
    -- Dados extras
    user_agent TEXT NULL,
    ip_address VARCHAR(45) NULL,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Chave estrangeira
    FOREIGN KEY (user_id) REFERENCES usuarios(id) ON DELETE SET NULL,
    
    -- Índices para performance
    INDEX idx_user_id (user_id),
    INDEX idx_click_id (click_id),
    INDEX idx_event_type (event_type),
    INDEX idx_created_at (created_at)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ================================================================
-- 9. TABELA DE LOG DE EVENTOS KWAI
-- ================================================================

CREATE TABLE IF NOT EXISTS kwai_events_log (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tracking_id INT NULL,
    
    -- Dados do evento
    event_name VARCHAR(100) NOT NULL,
    event_data JSON NULL,
    
    -- Status do envio
    status ENUM('pending', 'sent', 'failed') DEFAULT 'pending',
    response_data TEXT NULL,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    sent_at TIMESTAMP NULL,
    
    -- Chave estrangeira
    FOREIGN KEY (tracking_id) REFERENCES kwai_tracking(id) ON DELETE SET NULL,
    
    -- Índices para performance
    INDEX idx_tracking_id (tracking_id),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ================================================================
-- 10. INSERIR DADOS INICIAIS (opcional)
-- ================================================================

-- Inserir algumas palavras proibidas básicas
INSERT IGNORE INTO palavras_proibidas (palavra) VALUES 
('spam'),
('lixo'),
('merda'),
('porra'),
('caralho'),
('puto'),
('fdp'),
('vai se foder'),
('cu'),
('buceta');

-- ================================================================
-- 11. CRIAR USUÁRIO ADMIN PADRÃO (opcional)
-- ================================================================

-- Descomente as linhas abaixo para criar um usuário admin padrão
-- IMPORTANTE: Mude a senha após o primeiro login!

-- INSERT IGNORE INTO usuarios (nome, whatsapp, senha, is_member, saldo) VALUES 
-- ('Administrador', '00000000000', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 1, 0.00);
-- 
-- Senha padrão: "password" 
-- Lembre-se de alterar após o primeiro acesso!

-- ================================================================
-- 12. TABELA DE TOKENS DE LOGIN (Lembrar Login)
-- ================================================================

CREATE TABLE IF NOT EXISTS login_tokens (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    token VARCHAR(64) NOT NULL UNIQUE,
    expires_at DATETIME NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    INDEX idx_token (token),
    INDEX idx_user_id (user_id),
    INDEX idx_expires (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ================================================================
-- 13. OTIMIZAÇÕES E CONFIGURAÇÕES FINAIS
-- ================================================================

-- Otimizar tabelas
OPTIMIZE TABLE usuarios, conversas, mensagens, saldo_historico, saques;

-- Analisar tabelas para melhor performance
ANALYZE TABLE usuarios, conversas, mensagens, saldo_historico, saques;

-- ================================================================
-- FIM DO SCRIPT
-- ================================================================

-- Verificar se todas as tabelas foram criadas corretamente
SHOW TABLES;

-- Verificar estrutura da tabela principal
DESCRIBE usuarios;

-- Mensagem de sucesso
SELECT 'DATABASE SETUP COMPLETED SUCCESSFULLY!' as STATUS;