/**
 * ============================================================================
 * CONFIGURAÇÃO GLOBAL DO SISTEMA - FRONTEND
 * ============================================================================
 * 
 * Este arquivo centraliza todas as URLs e caminhos usados no JavaScript.
 * Ao transferir para outro domínio, ajuste apenas a variável BASE_PATH.
 * 
 * ============================================================================
 */

// ============================================================================
// CONFIGURAÇÃO PRINCIPAL
// ============================================================================

/**
 * BASE_PATH: Caminho base da aplicação (relativo à raiz do domínio)
 * 
 * Exemplos:
 * - Sistema na raiz: const BASE_PATH = '/';
 * - Sistema em subpasta: const BASE_PATH = '/mensagem/';
 * - Sistema em subpasta profunda: const BASE_PATH = '/app/chat/';
 * 
 * IMPORTANTE: Sempre termine com /
 */
const BASE_PATH = '/mensagem/';

// ============================================================================
// URLs AUTOMÁTICAS (NÃO EDITE)
// ============================================================================

const CONFIG = {
    // Caminho base
    basePath: BASE_PATH,
    
    // URLs de pastas
    backend: BASE_PATH + 'backend/',
    uploads: BASE_PATH + 'uploads/',
    assets: BASE_PATH + 'assets/',
    
    // Páginas principais
    pages: {
        autenticacao: BASE_PATH + 'autenticacao/index.html',
        principal: BASE_PATH + 'principal/index.html',
        chat: BASE_PATH + 'chat/index.html',
        perfil: BASE_PATH + 'perfil/index.html',
        saques: BASE_PATH + 'saques/index.html',
        completeProfile: BASE_PATH + 'complete_profile/index.html',
        painel: BASE_PATH + 'painel/index.html',
        liberacao1: BASE_PATH + 'liberacao1/index.html',
        liberacao2: BASE_PATH + 'liberacao2/index.html',
        liberacao3: BASE_PATH + 'liberacao3/index.html'
    },
    
    // Endpoints do backend
    api: {
        // Autenticação
        login: BASE_PATH + 'backend/login.php',
        register: BASE_PATH + 'backend/register.php',
        logout: BASE_PATH + 'backend/logout.php',
        checkSession: BASE_PATH + 'backend/check_session.php',
        checkAutoLogin: BASE_PATH + 'backend/check_auto_login.php',
        resetPassword: BASE_PATH + 'backend/reset_password.php',
        
        // Usuário
        getUserData: BASE_PATH + 'backend/get_user_data.php',
        getUserDataSimple: BASE_PATH + 'backend/get_user_data_simple.php',
        getUserProfile: BASE_PATH + 'backend/get_user_profile.php',
        updateProfile: BASE_PATH + 'backend/update_profile.php',
        updateProfileFixed: BASE_PATH + 'backend/update_profile_fixed.php',
        updateOnlineStatus: BASE_PATH + 'backend/update_online_status.php',
        setOffline: BASE_PATH + 'backend/set_offline.php',
        
        // Usuários online
        getOnlineUsers: BASE_PATH + 'backend/get_online_users.php',
        
        // Chat e mensagens
        getConversations: BASE_PATH + 'backend/get_conversations.php',
        getOrCreateConversation: BASE_PATH + 'backend/get_or_create_conversation.php',
        getMessages: BASE_PATH + 'backend/get_messages.php',
        sendMessage: BASE_PATH + 'backend/send_message.php',
        deleteMessage: BASE_PATH + 'backend/delete_message.php',
        checkNewMessages: BASE_PATH + 'backend/check_new_messages.php',
        checkReadStatus: BASE_PATH + 'backend/check_read_status.php',
        checkDeletedMessages: BASE_PATH + 'backend/check_deleted_messages.php',
        
        // Presentes
        sendGift: BASE_PATH + 'backend/send_gift.php',
        checkNewGifts: BASE_PATH + 'backend/check_new_gifts.php',
        
        // Bloqueios
        blockUser: BASE_PATH + 'backend/block_user.php',
        unblockUser: BASE_PATH + 'backend/unblock_user.php',
        checkBlockStatus: BASE_PATH + 'backend/check_block_status.php',
        
        // Saldo e Saques
        getSaldo: BASE_PATH + 'backend/get_saldo.php',
        savePix: BASE_PATH + 'backend/save_pix.php',
        requestWithdraw: BASE_PATH + 'backend/request_withdraw.php',
        getWithdrawals: BASE_PATH + 'backend/get_withdrawals.php',
        createBlockedWithdraw: BASE_PATH + 'backend/create_blocked_withdraw.php',
        checkTaxaStatus: BASE_PATH + 'backend/check_taxa_status.php',
        marcarTaxaPaga: BASE_PATH + 'backend/marcar_taxa_paga.php',
        liberarSaqueColaboracao: BASE_PATH + 'backend/liberar_saque_colaboracao.php',
        
        // Membership
        ativarMembership: BASE_PATH + 'backend/ativar_membership.php',
        
        // Admin
        adminLogin: BASE_PATH + 'backend/admin_login.php',
        adminStats: BASE_PATH + 'backend/admin_stats.php',
        adminGetUsers: BASE_PATH + 'backend/admin_get_users.php',
        adminGetConversations: BASE_PATH + 'backend/admin_get_conversations.php',
        adminGetMessages: BASE_PATH + 'backend/admin_get_messages.php',
        adminDeleteUser: BASE_PATH + 'backend/admin_delete_user.php',
        adminUpdateUser: BASE_PATH + 'backend/admin_update_user.php',
        adminUpdatePhoto: BASE_PATH + 'backend/admin_update_photo.php',
        adminClearMessages: BASE_PATH + 'backend/admin_clear_messages.php',
        
        // Kwai
        kwaiCaptureClickId: BASE_PATH + 'backend/kwai_capture_clickid.php',
        kwaiSendEvent: BASE_PATH + 'backend/kwai_send_event.php'
    },
    
    // Assets
    defaultLogo: BASE_PATH + 'assets/logo.png',
    moneySound: BASE_PATH + 'assets/money.mp3',
    
    // URLs de Pagamento (Configuráveis)
    // Edite aqui para mudar os links de checkout
    payment: {
        membership: 'https://pay.asuacomprasegura.com/checkout/dc17751d-f4cc-4ffc-a28b-a88438081436',
        taxa: 'https://pay.asuacomprasegura.com/checkout/8283b0f5-8d00-4cf4-920a-e833b1c96b82',
        colaboracao: 'https://pay.asuacomprasegura.com/checkout/dcffe9a3-ec35-42bd-9f6b-9fc63923d2d1'
    }
};

// ============================================================================
// FUNÇÕES AUXILIARES
// ============================================================================

/**
 * Retorna a URL completa para um arquivo de upload
 * @param {string} filename - Nome do arquivo
 * @returns {string} URL completa
 */
function getUploadUrl(filename) {
    if (!filename) {
        return CONFIG.defaultLogo;
    }
    
    // Se já é uma URL completa
    if (filename.startsWith('http://') || filename.startsWith('https://')) {
        return filename;
    }
    
    // Se já tem o BASE_PATH
    if (filename.startsWith(BASE_PATH)) {
        return filename;
    }
    
    // Se já tem uploads/
    if (filename.startsWith('uploads/')) {
        return BASE_PATH + filename;
    }
    
    // Caso contrário, adiciona uploads/
    return CONFIG.uploads + filename.replace(/^\//, '');
}

/**
 * Retorna a URL para um asset
 * @param {string} filename - Nome do arquivo
 * @returns {string} URL completa
 */
function getAssetUrl(filename) {
    if (!filename) {
        return CONFIG.defaultLogo;
    }
    
    if (filename.startsWith('http://') || filename.startsWith('https://')) {
        return filename;
    }
    
    if (filename.startsWith(BASE_PATH)) {
        return filename;
    }
    
    return CONFIG.assets + filename.replace(/^\//, '');
}

/**
 * Adiciona timestamp para evitar cache
 * @param {string} url - URL base
 * @returns {string} URL com timestamp
 */
function addTimestamp(url) {
    const separator = url.includes('?') ? '&' : '?';
    return url + separator + 'v=' + Date.now();
}

/**
 * Redireciona para uma página
 * @param {string} page - Nome da página (ex: 'principal', 'chat', 'autenticacao')
 */
function redirectTo(page) {
    if (CONFIG.pages[page]) {
        window.location.href = CONFIG.pages[page];
    } else {
        console.error('Página não encontrada:', page);
    }
}

// ============================================================================
// EXPORTAR PARA WINDOW (disponível globalmente)
// ============================================================================

window.CONFIG = CONFIG;
window.getUploadUrl = getUploadUrl;
window.getAssetUrl = getAssetUrl;
window.addTimestamp = addTimestamp;
window.redirectTo = redirectTo;

// ============================================================================
// FIM DA CONFIGURAÇÃO
// ============================================================================
