// Elementos DOM
const uploadArea = document.getElementById('uploadArea');
const photoInput = document.getElementById('photoInput');
const uploadPlaceholder = document.getElementById('uploadPlaceholder');
const imagePreview = document.getElementById('imagePreview');
const previewImage = document.getElementById('previewImage');
const removeImageBtn = document.getElementById('removeImage');
const profileForm = document.getElementById('profileForm');
const skipBtn = document.getElementById('skipBtn');

let selectedFile = null;

// ===== UPLOAD DE IMAGEM =====

// Click na área de upload
uploadArea.addEventListener('click', () => {
    photoInput.click();
});

// Prevenir comportamento padrão de drag
['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
    uploadArea.addEventListener(eventName, preventDefaults, false);
});

function preventDefaults(e) {
    e.preventDefault();
    e.stopPropagation();
}

// Highlight ao arrastar
['dragenter', 'dragover'].forEach(eventName => {
    uploadArea.addEventListener(eventName, () => {
        uploadArea.classList.add('dragover');
    });
});

['dragleave', 'drop'].forEach(eventName => {
    uploadArea.addEventListener(eventName, () => {
        uploadArea.classList.remove('dragover');
    });
});

// Drop de arquivo
uploadArea.addEventListener('drop', (e) => {
    const dt = e.dataTransfer;
    const files = dt.files;
    handleFile(files[0]);
});

// Seleção de arquivo via input
photoInput.addEventListener('change', (e) => {
    handleFile(e.target.files[0]);
});

// Processar arquivo
function handleFile(file) {
    if (!file) return;
    
    // Validar tipo
    const validTypes = ['image/jpeg', 'image/png', 'image/gif'];
    if (!validTypes.includes(file.type)) {
        showNotification('Apenas arquivos JPG, PNG ou GIF são permitidos', 'error');
        return;
    }
    
    // Validar tamanho (3MB = 3 * 1024 * 1024 bytes)
    const maxSize = 3 * 1024 * 1024;
    if (file.size > maxSize) {
        showNotification('A imagem deve ter no máximo 3MB', 'error');
        return;
    }
    
    selectedFile = file;
    
    // Mostrar preview
    const reader = new FileReader();
    reader.onload = (e) => {
        previewImage.src = e.target.result;
        uploadPlaceholder.classList.add('hidden');
        imagePreview.classList.remove('hidden');
    };
    reader.readAsDataURL(file);
}

// Remover imagem
removeImageBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    selectedFile = null;
    photoInput.value = '';
    uploadPlaceholder.classList.remove('hidden');
    imagePreview.classList.add('hidden');
    previewImage.src = '';
});

// ===== SUBMIT DO FORMULÁRIO =====

profileForm.addEventListener('submit', async (e) => {
    e.preventDefault();
    
    const idade = document.getElementById('idade').value;
    const sexo = document.getElementById('sexo').value;
    const submitBtn = profileForm.querySelector('.btn-submit');
    
    // Validações
    if (!selectedFile) {
        showNotification('Por favor, adicione uma foto de perfil', 'error');
        return;
    }
    
    if (!idade || idade < 16 || idade > 80) {
        showNotification('A idade deve estar entre 16 e 80 anos', 'error');
        return;
    }
    
    if (!sexo) {
        showNotification('Por favor, selecione seu sexo', 'error');
        return;
    }
    
    // Criar FormData para enviar arquivo
    const formData = new FormData();
    formData.append('foto_perfil', selectedFile);
    formData.append('idade', idade);
    formData.append('sexo', sexo);
    
    // Desabilitar botão e mostrar loading
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<span>Salvando...</span><i class="fas fa-spinner fa-spin"></i>';
    
    try {
        const response = await fetch('/mensagem/backend/update_profile.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        console.log('Response:', data);
        
        if (data.success) {
            showNotification(data.message, 'success');
            
            // Redirecionar para dashboard após 1.5s
            setTimeout(() => {
                window.location.href = '../principal/index.html';
            }, 1500);
        } else {
            showNotification(data.message, 'error');
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<span>Finalizar cadastro</span><i class="fas fa-arrow-right"></i>';
        }
        
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao salvar perfil', 'error');
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<span>Finalizar cadastro</span><i class="fas fa-arrow-right"></i>';
    }
});

// ===== PULAR ETAPA =====

skipBtn.addEventListener('click', () => {
    if (confirm('Tem certeza que deseja pular esta etapa? Você poderá completar seu perfil depois.')) {
        window.location.href = '../principal/index.html';
    }
});

// ===== SISTEMA DE NOTIFICAÇÕES =====

function showNotification(message, type = 'info') {
    // Remove existing notification
    const existing = document.querySelector('.notification');
    if (existing) {
        existing.remove();
    }
    
    // Create notification
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
        <span>${message}</span>
    `;
    
    // Add styles
    notification.style.cssText = `
        position: fixed;
        top: 24px;
        right: 24px;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
        color: white;
        padding: 16px 24px;
        border-radius: 12px;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        display: flex;
        align-items: center;
        gap: 12px;
        font-size: 14px;
        font-weight: 600;
        z-index: 9999;
        animation: slideInRight 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 3s
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

// Add animation styles
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from {
            transform: translateX(400px);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    @keyframes slideOutRight {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(400px);
            opacity: 0;
        }
    }
    
    .hidden {
        display: none !important;
    }
`;
document.head.appendChild(style);
