// Elementos DOM
const messagesContainer = document.getElementById('messagesContainer');
const messageInput = document.getElementById('messageInput');
const btnSend = document.getElementById('btnSend');
const btnAttach = document.getElementById('btnAttach');
const btnVoice = document.getElementById('btnVoice');
const otherUserPhoto = document.getElementById('otherUserPhoto');
const otherUserName = document.getElementById('otherUserName');
const otherUserStatus = document.getElementById('otherUserStatus');
const imageInput = document.getElementById('imageInput');
const videoInput = document.getElementById('videoInput');
const recordingIndicator = document.getElementById('recordingIndicator');
const recordingTime = document.getElementById('recordingTime');
const saldoDisplay = document.getElementById('saldoDisplay');
const saldoValue = document.getElementById('saldoValue');
const cooldownBar = document.getElementById('cooldownBar');
const moneyToast = document.getElementById('moneyToast');
const btnMenu = document.getElementById('btnMenu');
const menuDropdown = document.getElementById('menuDropdown');
const btnBlock = document.getElementById('btnBlock');
const btnGiftMenu = document.getElementById('btnGiftMenu');
const replyPreview = document.getElementById('replyPreview');
const replyPreviewName = document.getElementById('replyPreviewName');
const replyPreviewText = document.getElementById('replyPreviewText');
const replyPreviewClose = document.getElementById('replyPreviewClose');
const membershipModal = document.getElementById('membershipModal');
const btnUpgrade = document.getElementById('btnUpgrade');
const btnCancelMembership = document.getElementById('btnCancelMembership');
const quickReplies = document.getElementById('quickReplies');
const giftModal = document.getElementById('giftModal');
const giftCurrentBalance = document.getElementById('giftCurrentBalance');
const giftCustomAmount = document.getElementById('giftCustomAmount');
const giftMessage = document.getElementById('giftMessage');
const giftCharCount = document.getElementById('giftCharCount');
const btnSendGift = document.getElementById('btnSendGift');

// Áudio de ganho
const moneySound = new Audio('/mensagem/assets/money.mp3');

// Variáveis globais
let conversaId = null;
let otherUserId = null;
let lastMessageId = 0;
let canEarnMoney = true;
let cooldownTimer = null;
let messagesInterval = null;
let onlineInterval = null;
let giftsCheckInterval = null;
let notifiedGifts = new Set();
let userBlocked = false; // Controle se o usuário atingiu o limite
let isSending = false; // Controle para eviar envios duplicados
let hasInteraction = false; // Controle se já houve interação no chat
let otherUserData = {
    nome: '',
    idade: null,
    sexo: '',
    foto_perfil: '',
    online: false
};
let selectedMedia = null;
let selectedMediaType = null;
let replyingTo = null; // Mensagem sendo respondida
let currentGiftAmount = 0; // Valor do presente selecionado
let currentBalance = 0; // Saldo atual do usuário

// Variáveis de gravação de áudio
let mediaRecorder = null;
let audioChunks = [];
let audioStream = null;
let recordingStartTime = null;
let recordingTimer = null;
let isRecording = false;

// Prevenir zoom em dispositivos móveis
document.addEventListener('gesturestart', function(e) {
    e.preventDefault();
});

document.addEventListener('gesturechange', function(e) {
    e.preventDefault();
});

document.addEventListener('gestureend', function(e) {
    e.preventDefault();
});

// Prevenir zoom com duplo toque
let lastTouchEnd = 0;
document.addEventListener('touchend', function(event) {
    const now = Date.now();
    if (now - lastTouchEnd <= 300) {
        event.preventDefault();
    }
    lastTouchEnd = now;
}, false);

// Prevenir zoom com roda do mouse (desktop)
document.addEventListener('wheel', function(e) {
    if (e.ctrlKey) {
        e.preventDefault();
    }
}, { passive: false });

// Prevenir zoom com teclado (Ctrl + / Ctrl -)
document.addEventListener('keydown', function(e) {
    if ((e.ctrlKey || e.metaKey) && (e.key === '+' || e.key === '-' || e.key === '=')) {
        e.preventDefault();
    }
}, false);

// Atualizar status online
async function updateOnlineStatus() {
    try {
        await fetch('/mensagem/backend/update_online_status.php');
    } catch (error) {
        console.error('Erro ao atualizar status:', error);
    }
}

// Obter parâmetros da URL
const urlParams = new URLSearchParams(window.location.search);
const userId = urlParams.get('user_id');

if (!userId) {
    window.location.href = '/mensagem/principal/index.html';
}

// Iniciar conversa
async function initChat() {
    try {
        const formData = new FormData();
        formData.append('user_id', userId);
        
        const response = await fetch('/mensagem/backend/get_or_create_conversation.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        console.log('Resposta da API:', data); // Debug
        
        if (data.success) {
            conversaId = data.conversa_id;
            otherUserId = data.other_user.id;
            
            // Armazenar dados do outro usuário
            otherUserData = {
                nome: data.other_user.nome,
                idade: data.other_user.idade,
                sexo: data.other_user.sexo,
                foto_perfil: data.other_user.foto_perfil,
                online: data.other_user.online
            };
            
            // Atualizar info do usuário (apenas primeiro nome)
            const primeiroNome = data.other_user.nome.split(' ')[0];
            otherUserName.textContent = primeiroNome;
            if (data.other_user.foto_perfil) {
                otherUserPhoto.src = `/mensagem/${data.other_user.foto_perfil}`;
            }
            
            // Atualizar status inicial
            if (data.other_user.online) {
                otherUserStatus.textContent = 'Online';
                otherUserStatus.className = 'chat-user-status online';
            } else {
                otherUserStatus.textContent = 'Offline';
                otherUserStatus.className = 'chat-user-status offline';
            }
            
            // Tornar avatar clicável
            otherUserPhoto.style.cursor = 'pointer';
            otherUserPhoto.onclick = function() {
                openProfileModal();
            };
            
            // Carregar mensagens
            await loadMessages();
            
            // Carregar saldo
            await loadSaldo();
            
            // Verificar status de bloqueio
            await checkBlockStatus();
            
            // Iniciar polling
            startMessagesPolling();
        } else {
            console.error('Erro da API:', data.message);
            showNotification(data.message || 'Erro ao iniciar conversa', 'error');
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao carregar chat', 'error');
    }
}

// Carregar mensagens
async function loadMessages(isPolling = false) {
    try {
        const url = lastMessageId > 0 
            ? `/mensagem/backend/get_messages.php?conversa_id=${conversaId}&last_id=${lastMessageId}`
            : `/mensagem/backend/get_messages.php?conversa_id=${conversaId}`;
        
        const response = await fetch(url);
        const data = await response.json();
        
        if (data.success && data.mensagens.length > 0) {
            // Verificar se já houve interação
            if (data.mensagens.length > 0) {
                hasInteraction = true;
                hideQuickReplies();
            }
            
            data.mensagens.forEach(msg => {
                addMessageToUI(msg, isPolling);
                lastMessageId = Math.max(lastMessageId, msg.id);
            });
            
            if (!isPolling) {
                scrollToBottom();
            }
        } else if (data.success && data.mensagens.length === 0) {
            // Sem mensagens, mostrar atalhos
            showQuickReplies();
        }
    } catch (error) {
        console.error('Erro ao carregar mensagens:', error);
    }
}

// Adicionar mensagem na UI
function addMessageToUI(msg, animate = false) {
    // Verificar se a mensagem já existe no DOM (evitar duplicatas)
    const existingMessage = messagesContainer.querySelector(`.message[data-message-id="${msg.id}"]`);
    if (existingMessage) {
        return existingMessage; // Já existe, não adicionar novamente
    }
    
    const messageDiv = document.createElement('div');
    messageDiv.className = `message ${msg.is_mine ? 'mine' : 'other'}`;
    messageDiv.setAttribute('data-message-id', msg.id);
    
    const bubbleDiv = document.createElement('div');
    bubbleDiv.className = 'message-bubble';
    
    // Se é uma mensagem de presente
    if (msg.tipo === 'gift' || msg.tipo_midia === 'gift') {
        const giftDiv = document.createElement('div');
        giftDiv.className = 'message-gift';
        giftDiv.innerHTML = `
            <div class="gift-bubble-icon">🎁</div>
            <div class="gift-bubble-details">
                <div class="gift-bubble-amount">R$ ${parseFloat(msg.valor || msg.gift_amount || 0).toFixed(2).replace('.', ',')}</div>
                <div class="gift-bubble-label">Presente ${msg.is_mine ? 'enviado' : 'recebido'}</div>
                ${msg.mensagem && msg.mensagem.trim() ? `<div class="gift-bubble-message">${msg.mensagem}</div>` : ''}
            </div>
        `;
        bubbleDiv.appendChild(giftDiv);
    } else if (msg.tipo_midia && msg.caminho_midia) {
        // Se tem mídia, adicionar preview
        const mediaDiv = document.createElement('div');
        mediaDiv.className = 'message-media';
        
        if (msg.tipo_midia === 'image') {
            const img = document.createElement('img');
            img.src = `/mensagem/${msg.caminho_midia}`;
            img.alt = 'Imagem enviada';
            img.onclick = () => openImageViewer(`/mensagem/${msg.caminho_midia}`);
            img.style.cursor = 'pointer';
            mediaDiv.appendChild(img);
        } else if (msg.tipo_midia === 'video') {
            mediaDiv.classList.add('video-media');
            const video = document.createElement('video');
            video.src = `/mensagem/${msg.caminho_midia}`;
            video.preload = 'metadata';
            video.setAttribute('playsinline', ''); // Importante para iOS
            video.muted = true; // Necessário para autoplay em alguns navegadores
            
            // Tentar carregar primeiro frame
            video.addEventListener('loadeddata', function() {
                video.currentTime = 0.1;
            });
            
            mediaDiv.appendChild(video);
            
            // Ao clicar, abrir em fullscreen
            mediaDiv.onclick = () => openVideoPlayer(`/mensagem/${msg.caminho_midia}`);
        } else if (msg.tipo_midia === 'audio') {
            mediaDiv.classList.add('audio-media');
            
            // Criar player de áudio customizado
            const audioPlayer = document.createElement('div');
            audioPlayer.className = 'audio-player';
            
            const playBtn = document.createElement('button');
            playBtn.className = 'audio-play-btn';
            playBtn.innerHTML = '<i class="fas fa-play"></i>';
            
            const waveform = document.createElement('div');
            waveform.className = 'audio-waveform';
            
            // Criar barras de onda decorativas
            for (let i = 0; i < 30; i++) {
                const bar = document.createElement('div');
                bar.className = 'audio-bar';
                bar.style.height = `${Math.random() * 70 + 30}%`;
                waveform.appendChild(bar);
            }
            
            const duration = document.createElement('span');
            duration.className = 'audio-duration';
            duration.textContent = '...';
            
            const audio = document.createElement('audio');
            audio.src = `/mensagem/${msg.caminho_midia}`;
            audio.preload = 'metadata';
            
            // Atualizar duração quando carregar
            audio.addEventListener('loadedmetadata', () => {
                if (isFinite(audio.duration)) {
                    const mins = Math.floor(audio.duration / 60);
                    const secs = Math.floor(audio.duration % 60);
                    duration.textContent = `${mins}:${secs.toString().padStart(2, '0')}`;
                }
            });
            
            // Fallback se loadedmetadata não disparar
            audio.addEventListener('canplay', () => {
                if (duration.textContent === '...' && isFinite(audio.duration)) {
                    const mins = Math.floor(audio.duration / 60);
                    const secs = Math.floor(audio.duration % 60);
                    duration.textContent = `${mins}:${secs.toString().padStart(2, '0')}`;
                }
            });
            
            // Atualizar tempo durante reprodução
            audio.addEventListener('timeupdate', () => {
                if (isFinite(audio.duration) && isFinite(audio.currentTime)) {
                    const currentMins = Math.floor(audio.currentTime / 60);
                    const currentSecs = Math.floor(audio.currentTime % 60);
                    const totalMins = Math.floor(audio.duration / 60);
                    const totalSecs = Math.floor(audio.duration % 60);
                    duration.textContent = `${currentMins}:${currentSecs.toString().padStart(2, '0')} / ${totalMins}:${totalSecs.toString().padStart(2, '0')}`;
                }
            });
            
            // Play/pause
            playBtn.onclick = () => {
                if (audio.paused) {
                    // Pausar todos os outros áudios
                    document.querySelectorAll('.audio-player audio').forEach(a => {
                        if (a !== audio) a.pause();
                    });
                    document.querySelectorAll('.audio-play-btn').forEach(btn => {
                        btn.innerHTML = '<i class="fas fa-play"></i>';
                    });
                    
                    audio.play();
                    playBtn.innerHTML = '<i class="fas fa-pause"></i>';
                } else {
                    audio.pause();
                    playBtn.innerHTML = '<i class="fas fa-play"></i>';
                }
            };
            
            // Reset ao terminar
            audio.addEventListener('ended', () => {
                playBtn.innerHTML = '<i class="fas fa-play"></i>';
                audio.currentTime = 0;
                if (isFinite(audio.duration)) {
                    const mins = Math.floor(audio.duration / 60);
                    const secs = Math.floor(audio.duration % 60);
                    duration.textContent = `${mins}:${secs.toString().padStart(2, '0')}`;
                }
            });
            
            // Pausar quando pausar
            audio.addEventListener('pause', () => {
                playBtn.innerHTML = '<i class="fas fa-play"></i>';
            });
            
            audioPlayer.appendChild(playBtn);
            audioPlayer.appendChild(waveform);
            audioPlayer.appendChild(duration);
            audioPlayer.appendChild(audio);
            
            mediaDiv.appendChild(audioPlayer);
        }
        
        bubbleDiv.appendChild(mediaDiv);
    }
    
    // Mensagem respondida (se houver)
    if (msg.reply_to_id && msg.replied_message) {
        const repliedDiv = document.createElement('div');
        repliedDiv.className = 'replied-message';
        
        const repliedName = document.createElement('div');
        repliedName.className = 'replied-message-name';
        repliedName.textContent = msg.replied_message.is_mine ? 'Você' : otherUserData.nome.split(' ')[0];
        
        const repliedText = document.createElement('div');
        repliedText.className = 'replied-message-text';
        
        if (msg.replied_message.tipo_midia === 'gift' || msg.replied_message.tipo === 'gift') {
            const valor = parseFloat(msg.replied_message.gift_amount || msg.replied_message.valor || 0).toFixed(2).replace('.', ',');
            repliedText.textContent = `🎁 Presente R$ ${valor}`;
        } else if (msg.replied_message.tipo_midia === 'image') {
            repliedText.textContent = '📷 Imagem';
        } else if (msg.replied_message.tipo_midia === 'video') {
            repliedText.textContent = '🎥 Vídeo';
        } else if (msg.replied_message.tipo_midia === 'audio') {
            repliedText.textContent = '🎤 Áudio';
        } else {
            repliedText.textContent = msg.replied_message.mensagem || '';
        }
        
        repliedDiv.appendChild(repliedName);
        repliedDiv.appendChild(repliedText);
        bubbleDiv.appendChild(repliedDiv);
    }
    
    // Texto da mensagem (se houver e se não for presente)
    if (msg.mensagem && msg.mensagem.trim() && msg.tipo_midia !== 'gift' && msg.tipo !== 'gift') {
        const textDiv = document.createElement('div');
        textDiv.className = 'message-text';
        textDiv.textContent = msg.mensagem;
        bubbleDiv.appendChild(textDiv);
    }
    
    const timeDiv = document.createElement('div');
    timeDiv.className = 'message-time';
    
    // Adicionar ícone de status apenas para mensagens enviadas por mim
    let statusIcon = '';
    if (msg.is_mine) {
        if (msg.lida) {
            statusIcon = '<i class="fas fa-check-double status-icon read"></i>';
        } else {
            statusIcon = '<i class="fas fa-check-double status-icon sent"></i>';
        }
    }
    
    timeDiv.innerHTML = `${msg.time} ${statusIcon}`;
    
    bubbleDiv.appendChild(timeDiv);
    
    // Criar wrapper para bubble + ícone de opções
    const bubbleWrapper = document.createElement('div');
    bubbleWrapper.className = 'bubble-wrapper';
    bubbleWrapper.appendChild(bubbleDiv);
    
    // Adicionar ícone de opções (3 pontinhos) apenas para minhas mensagens
    if (msg.is_mine) {
        const optionsIcon = document.createElement('div');
        optionsIcon.className = 'message-options';
        optionsIcon.innerHTML = '<i class="fas fa-ellipsis-v"></i>';
        optionsIcon.onclick = (e) => {
            e.stopPropagation();
            showDeleteOption(messageDiv, msg);
        };
        bubbleWrapper.appendChild(optionsIcon);
    }
    
    messageDiv.appendChild(bubbleWrapper);
    messagesContainer.appendChild(messageDiv);
    
    // Adicionar funcionalidade de swipe para responder
    addSwipeToMessage(messageDiv, msg);
    
    if (animate) {
        scrollToBottom();
        
        // Se não é minha mensagem, mostrar toast
        if (!msg.is_mine) {
            playNotificationSound();
        }
    }
    
    return messageDiv;
}

// Enviar mensagem
async function sendMessage() {
    const mensagem = messageInput.value.trim();
    
    if (!mensagem || !otherUserId || isSending) return;
    
    // Marcar que está enviando
    isSending = true;
    
    // Marcar que houve interação e esconder atalhos
    hasInteraction = true;
    hideQuickReplies();
    
    const formData = new FormData();
    formData.append('destinatario_id', otherUserId);
    formData.append('mensagem', mensagem);
    
    // Se está respondendo a uma mensagem
    let repliedMessageData = null;
    if (replyingTo) {
        formData.append('reply_to_id', replyingTo.id);
        // Guardar dados da mensagem respondida antes de limpar
        repliedMessageData = {
            id: replyingTo.id,
            mensagem: replyingTo.mensagem,
            tipo_midia: replyingTo.tipo_midia,
            remetente_id: replyingTo.remetente_id,
            remetente_nome: replyingTo.remetente_nome
        };
    }
    
    // Limpar input imediatamente
    messageInput.value = '';
    
    // Esconder preview de resposta
    replyPreview.style.display = 'none';
    replyingTo = null;
    
    try {
        const response = await fetch('/mensagem/backend/send_message.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            // Adicionar mensagem enviada
            const messageElement = addMessageToUI({
                id: data.mensagem_id,
                mensagem: mensagem,
                is_mine: true,
                time: new Date().toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' }),
                lida: false,
                reply_to_id: repliedMessageData ? repliedMessageData.id : null,
                replied_message: repliedMessageData
            }, true);
            
            lastMessageId = Math.max(lastMessageId, data.mensagem_id);
            
            // Atualizar saldo e mostrar efeito apenas se pode ganhar
            if (data.saldo && data.ganhou) {
                saldoValue.textContent = `R$ ${data.saldo}`;
                showMoneyToast(data.ganho);
                startCooldown();
            } else if (data.saldo) {
                // Apenas atualiza o saldo sem efeitos
                saldoValue.textContent = `R$ ${data.saldo}`;
            }
        } else {
            // Verificar se foi banido e excluído por palavras proibidas
            if (data.banido && data.excluido) {
                showNotification('🚫 ' + (data.mensagem || 'Sua conta foi excluída por violar as regras.'), 'error');
                setTimeout(() => {
                    redirectTo('autenticacao');
                }, 1500);
                return;
            } else if (data.banido) {
                showNotification('🚫 ' + (data.mensagem || 'Você foi banido por violar as regras.'), 'error');
                setTimeout(() => {
                    redirectTo('autenticacao');
                }, 2000);
                return;
            }
            
            // Verificar se precisa de membership
            if (data.requires_membership) {
                userBlocked = true; // Marcar usuário como bloqueado
                showMembershipModal();
                // Restaurar a mensagem no input
                messageInput.value = mensagem;
            } else {
                // Mostrar aviso de palavra proibida
                showNotification(data.message || 'Erro ao enviar mensagem', 'error');
                // Restaurar a mensagem no input para tentar editar
                messageInput.value = mensagem;
            }
        }
    } catch (error) {
        console.error('Erro:', error);
        showNotification('Erro ao enviar mensagem', 'error');
    } finally {
        // Liberar envio após conclusão (sucesso ou erro)
        isSending = false;
    }
}

// Polling de mensagens
// Verificar novos presentes recebidos no chat
async function checkNewGiftsInChat() {
    try {
        const response = await fetch('/mensagem/backend/check_new_gifts.php');
        const data = await response.json();
        
        if (data.success && data.novos_presentes.length > 0) {
            data.novos_presentes.forEach(gift => {
                if (!notifiedGifts.has(gift.id)) {
                    console.log('Novo presente recebido no chat:', gift);
                    notifiedGifts.add(gift.id);
                    
                    // Atualizar saldo na interface
                    saldoValue.textContent = `R$ ${data.saldo_atual}`;
                    currentBalance = parseFloat(data.saldo_atual);
                    
                    // Se o presente é da conversa atual, recarregar mensagens
                    if (gift.remetente_id == otherUserId) {
                        loadMessages(true);
                    }
                }
            });
        }
    } catch (error) {
        console.error('Erro ao verificar presentes:', error);
    }
}

function startMessagesPolling() {
    messagesInterval = setInterval(() => {
        loadMessages(true);
    }, 3000); // Verifica novas mensagens a cada 3 segundos
    
    // Atualizar status online do próprio usuário
    updateOnlineStatus();
    onlineInterval = setInterval(updateOnlineStatus, 30000);
    
    // Verificar status online do outro usuário
    checkOtherUserStatus();
    setInterval(checkOtherUserStatus, 5000); // Verifica a cada 5 segundos
    
    // Verificar status de leitura das mensagens
    checkReadStatus();
    setInterval(checkReadStatus, 3000); // Verifica a cada 3 segundos
    
    // Verificar mensagens deletadas
    checkDeletedMessages();
    setInterval(checkDeletedMessages, 3000); // Verifica a cada 3 segundos
    
    // Verificar novos presentes recebidos
    checkNewGiftsInChat();
    giftsCheckInterval = setInterval(checkNewGiftsInChat, 3000);
    
    // Atualizar saldo periodicamente
    loadSaldo();
    setInterval(loadSaldo, 5000); // Atualiza saldo a cada 5 segundos
}

// Verificar status online do outro usuário
async function checkOtherUserStatus() {
    if (!otherUserId) return;
    
    try {
        const response = await fetch(`/mensagem/backend/check_user_status.php?user_id=${otherUserId}`);
        const data = await response.json();
        
        if (data.success) {
            const isOnline = data.online;
            otherUserData.online = isOnline;
            
            // Atualizar texto do status
            if (isOnline) {
                otherUserStatus.textContent = 'Online';
                otherUserStatus.className = 'chat-user-status online';
            } else {
                otherUserStatus.textContent = 'Offline';
                otherUserStatus.className = 'chat-user-status offline';
            }
        }
    } catch (error) {
        console.error('Erro ao verificar status:', error);
    }
}

// Verificar status de leitura das mensagens
async function checkReadStatus() {
    if (!conversaId) return;
    
    try {
        const response = await fetch(`/mensagem/backend/check_read_status.php?conversa_id=${conversaId}`);
        const data = await response.json();
        
        if (data.success && data.read_messages.length > 0) {
            // Atualizar os checkmarks das mensagens lidas
            data.read_messages.forEach(msg => {
                const messageElement = messagesContainer.querySelector(`.message[data-message-id="${msg.id}"]`);
                if (messageElement) {
                    const statusIcon = messageElement.querySelector('.status-icon');
                    if (statusIcon && statusIcon.classList.contains('sent')) {
                        statusIcon.classList.remove('sent');
                        statusIcon.classList.add('read');
                    }
                }
            });
        }
    } catch (error) {
        console.error('Erro ao verificar status de leitura:', error);
    }
}

// Verificar mensagens deletadas
async function checkDeletedMessages() {
    if (!conversaId) return;
    
    // Coletar IDs de todas as mensagens no DOM
    const messageElements = messagesContainer.querySelectorAll('.message[data-message-id]');
    if (messageElements.length === 0) return;
    
    const messageIds = Array.from(messageElements).map(el => el.getAttribute('data-message-id')).join(',');
    
    try {
        const response = await fetch(`/mensagem/backend/check_deleted_messages.php?conversa_id=${conversaId}&message_ids=${messageIds}`);
        const data = await response.json();
        
        if (data.success && data.deleted.length > 0) {
            // Remover mensagens deletadas do DOM
            data.deleted.forEach(messageId => {
                const messageElement = messagesContainer.querySelector(`.message[data-message-id="${messageId}"]`);
                if (messageElement) {
                    // Animar remoção
                    messageElement.style.opacity = '0';
                    messageElement.style.transform = 'scale(0.8)';
                    messageElement.style.transition = 'all 0.3s ease';
                    
                    setTimeout(() => {
                        messageElement.remove();
                    }, 300);
                }
            });
        }
    } catch (error) {
        console.error('Erro ao verificar mensagens deletadas:', error);
    }
}

// Scroll para o final
function scrollToBottom() {
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
}

// Som de notificação
function playNotificationSound() {
    // Criar elemento de áudio (você pode adicionar um arquivo de som depois)
    const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBTGH0fPTgjMGHm7A7+OZURE');
    audio.volume = 0.3;
    audio.play().catch(() => {}); // Ignorar erro se não puder tocar
}

// Event listeners
btnSend.addEventListener('click', sendMessage);

messageInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
        sendMessage();
    }
});

// Bloquear input se usuário atingiu limite
messageInput.addEventListener('focus', (e) => {
    if (userBlocked) {
        e.target.blur(); // Remove o foco, evitando o teclado
        showMembershipModal();
    }
});

messageInput.addEventListener('click', (e) => {
    if (userBlocked) {
        e.preventDefault();
        showMembershipModal();
    }
});

// Limpar interval ao sair
window.addEventListener('beforeunload', () => {
    if (messagesInterval) {
        clearInterval(messagesInterval);
    }
    if (onlineInterval) {
        clearInterval(onlineInterval);
    }
});

// Notificações
function showNotification(message, type = 'info') {
    const existing = document.querySelector('.notification');
    if (existing) existing.remove();
    
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
        <span>${message}</span>
    `;
    
    notification.style.cssText = `
        position: fixed;
        top: 90px;
        right: 24px;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
        color: white;
        padding: 16px 24px;
        border-radius: 12px;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        display: flex;
        align-items: center;
        gap: 12px;
        font-size: 14px;
        font-weight: 600;
        z-index: 9999;
        animation: slideInRight 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

// Funções do Modal de Perfil
function openProfileModal() {
    const modal = document.getElementById('profileModal');
    const modalPhoto = document.getElementById('modalUserPhoto');
    const modalName = document.getElementById('modalUserName');
    const modalAge = document.getElementById('modalUserAge');
    const modalGender = document.getElementById('modalUserGender');
    const modalOnlineDot = document.querySelector('.modal-online-dot');
    
    // Preencher dados
    modalName.textContent = otherUserData.nome;
    
    // Idade
    if (otherUserData.idade) {
        modalAge.textContent = otherUserData.idade + ' anos';
    } else {
        modalAge.textContent = 'Não informado';
    }
    
    // Sexo - suporta múltiplos formatos
    let sexoTexto = 'Não informado';
    if (otherUserData.sexo) {
        const sexo = otherUserData.sexo.toLowerCase();
        if (sexo === 'masculino' || sexo === 'm') {
            sexoTexto = 'Masculino';
        } else if (sexo === 'feminino' || sexo === 'f') {
            sexoTexto = 'Feminino';
        } else if (sexo === 'outro') {
            sexoTexto = 'Outro';
        }
    }
    modalGender.textContent = sexoTexto;
    
    if (otherUserData.foto_perfil) {
        modalPhoto.src = `/mensagem/${otherUserData.foto_perfil}`;
    } else {
        modalPhoto.src = '/mensagem/assets/logo.png';
    }
    
    // Status online
    if (modalOnlineDot) {
        modalOnlineDot.style.display = otherUserData.online ? 'block' : 'none';
    }
    
    // Mostrar modal
    modal.style.display = 'flex';
    document.body.style.overflow = 'hidden';
}

function closeProfileModal() {
    const modal = document.getElementById('profileModal');
    modal.style.display = 'none';
    document.body.style.overflow = 'auto';
}

// Event listeners para fechar modal
document.addEventListener('DOMContentLoaded', () => {
    const modal = document.getElementById('profileModal');
    const closeBtn = document.querySelector('.profile-modal-close');
    const modalOverlay = modal;
    
    // Fechar ao clicar no X
    if (closeBtn) {
        closeBtn.addEventListener('click', closeProfileModal);
    }
    
    // Fechar ao clicar fora do modal
    if (modalOverlay) {
        modalOverlay.addEventListener('click', (e) => {
            if (e.target === modalOverlay) {
                closeProfileModal();
            }
        });
    }
    
    // Fechar com ESC
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
            const modal = document.getElementById('profileModal');
            if (modal && modal.style.display === 'flex') {
                closeProfileModal();
            }
            const mediaModal = document.getElementById('mediaModal');
            if (mediaModal && mediaModal.style.display === 'flex') {
                closeMediaModal();
            }
        }
    });
});

// ===== SISTEMA DE MÍDIA =====

// Abrir modal de mídia
btnAttach.addEventListener('click', () => {
    document.getElementById('mediaModal').style.display = 'flex';
});

// Fechar modal de mídia
function closeMediaModal() {
    document.getElementById('mediaModal').style.display = 'none';
    cancelMedia();
}

// Handler para imagem
imageInput.addEventListener('change', (e) => {
    const file = e.target.files[0];
    if (!file) return;
    
    // Validar tamanho (3MB)
    if (file.size > 3 * 1024 * 1024) {
        showNotification('Imagem muito grande. Máximo 3MB', 'error');
        imageInput.value = '';
        return;
    }
    
    // Validar tipo
    if (!file.type.startsWith('image/')) {
        showNotification('Arquivo não é uma imagem válida', 'error');
        imageInput.value = '';
        return;
    }
    
    selectedMedia = file;
    selectedMediaType = 'image';
    showPreview(file, 'image');
});

// Handler para vídeo
videoInput.addEventListener('change', (e) => {
    const file = e.target.files[0];
    if (!file) return;
    
    // Validar tamanho (50MB)
    if (file.size > 50 * 1024 * 1024) {
        showNotification('Vídeo muito grande. Máximo 50MB', 'error');
        videoInput.value = '';
        return;
    }
    
    // Validar tipo
    if (!file.type.startsWith('video/')) {
        showNotification('Arquivo não é um vídeo válido', 'error');
        videoInput.value = '';
        return;
    }
    
    selectedMedia = file;
    selectedMediaType = 'video';
    
    // Validar duração
    const video = document.createElement('video');
    video.preload = 'metadata';
    
    video.onloadedmetadata = function() {
        window.URL.revokeObjectURL(video.src);
        if (video.duration > 60) {
            showNotification('Vídeo muito longo. Máximo 1 minuto', 'error');
            videoInput.value = '';
            selectedMedia = null;
            selectedMediaType = null;
        } else {
            showPreview(file, 'video');
        }
    };
    
    video.src = URL.createObjectURL(file);
});

// Mostrar preview
function showPreview(file, type) {
    const uploadOptions = document.querySelector('.media-upload-options');
    const preview = document.getElementById('mediaPreview');
    const previewImage = document.getElementById('previewImage');
    const previewVideo = document.getElementById('previewVideo');
    
    uploadOptions.style.display = 'none';
    preview.style.display = 'block';
    
    const url = URL.createObjectURL(file);
    
    if (type === 'image') {
        previewImage.src = url;
        previewImage.style.display = 'block';
        previewVideo.style.display = 'none';
    } else {
        previewVideo.src = url;
        previewVideo.style.display = 'block';
        previewImage.style.display = 'none';
    }
}

// Cancelar mídia
function cancelMedia() {
    const uploadOptions = document.querySelector('.media-upload-options');
    const preview = document.getElementById('mediaPreview');
    const previewImage = document.getElementById('previewImage');
    const previewVideo = document.getElementById('previewVideo');
    const progressContainer = document.getElementById('uploadProgress');
    const progressBar = document.getElementById('uploadProgressBar');
    const progressText = document.getElementById('uploadProgressText');
    
    uploadOptions.style.display = 'grid';
    preview.style.display = 'none';
    previewImage.src = '';
    previewVideo.src = '';
    
    // Resetar barra de progresso
    progressContainer.style.display = 'none';
    progressBar.style.width = '0%';
    progressText.textContent = 'Enviando...';
    
    selectedMedia = null;
    selectedMediaType = null;
    imageInput.value = '';
    videoInput.value = '';
}

// Enviar mídia
async function sendMedia() {
    if (!selectedMedia || !selectedMediaType) return;
    
    const sendBtn = document.querySelector('.btn-send-media');
    const progressContainer = document.getElementById('uploadProgress');
    const progressBar = document.getElementById('uploadProgressBar');
    const progressText = document.getElementById('uploadProgressText');
    
    sendBtn.disabled = true;
    sendBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processando...';
    
    try {
        // Comprimir mídia antes de enviar
        let mediaToUpload = selectedMedia;
        
        if (selectedMediaType === 'image') {
            progressContainer.style.display = 'block';
            progressBar.style.width = '0%';
            progressText.textContent = 'Comprimindo imagem...';
            mediaToUpload = await compressImage(selectedMedia);
        }
        
        // Para vídeos, não comprimir para preservar áudio
        // A validação de tamanho será feita no backend
        
        // Mostrar barra de progresso
        progressContainer.style.display = 'block';
        progressText.textContent = 'Enviando...';

        
        // 1. Fazer upload da mídia com progresso
        const formData = new FormData();
        formData.append('media', mediaToUpload);
        formData.append('type', selectedMediaType);
        
        const uploadData = await uploadWithProgress(formData, (percent) => {
            progressBar.style.width = percent + '%';
            progressText.textContent = `Enviando... ${percent}%`;
        });
        
        if (!uploadData.success) {
            showNotification(uploadData.message || 'Erro ao enviar mídia', 'error');
            progressContainer.style.display = 'none';
            sendBtn.disabled = false;
            sendBtn.innerHTML = '<i class="fas fa-paper-plane"></i> Enviar';
            return;
        }
        
        // 2. Enviar mensagem com a mídia
        progressText.textContent = 'Finalizando...';
        const messageFormData = new FormData();
        messageFormData.append('destinatario_id', otherUserId);
        messageFormData.append('mensagem', ''); // Sem texto automático
        messageFormData.append('tipo_midia', selectedMediaType);
        messageFormData.append('caminho_midia', uploadData.path);
        
        const messageResponse = await fetch('/mensagem/backend/send_message.php', {
            method: 'POST',
            body: messageFormData
        });
        
        const messageData = await messageResponse.json();
        
        if (messageData.success) {
            progressText.textContent = 'Enviado!';
            
            // Atualizar saldo e mostrar efeito apenas se pode ganhar
            if (messageData.saldo && messageData.ganhou) {
                saldoValue.textContent = `R$ ${messageData.saldo}`;
                showMoneyToast();
                startCooldown();
            } else if (messageData.saldo) {
                saldoValue.textContent = `R$ ${messageData.saldo}`;
            }
            
            setTimeout(() => {
                closeMediaModal();
                loadMessages(true);
            }, 500);
        } else {
            showNotification(messageData.message || 'Erro ao enviar mensagem', 'error');
            progressContainer.style.display = 'none';
        }
        
    } catch (error) {
        console.error('Erro ao enviar mídia:', error);
        showNotification('Erro ao enviar mídia', 'error');
        progressContainer.style.display = 'none';
    } finally {
        sendBtn.disabled = false;
        sendBtn.innerHTML = '<i class="fas fa-paper-plane"></i> Enviar';
    }
}

// Comprimir imagem usando Canvas
async function compressImage(file) {
    return new Promise((resolve, reject) => {
        const reader = new FileReader();
        
        reader.onload = (e) => {
            const img = new Image();
            
            img.onload = () => {
                const canvas = document.createElement('canvas');
                const ctx = canvas.getContext('2d');
                
                // Redimensionar se muito grande
                let width = img.width;
                let height = img.height;
                const maxDimension = 1920; // Máximo 1920px
                
                if (width > maxDimension || height > maxDimension) {
                    if (width > height) {
                        height = (height / width) * maxDimension;
                        width = maxDimension;
                    } else {
                        width = (width / height) * maxDimension;
                        height = maxDimension;
                    }
                }
                
                canvas.width = width;
                canvas.height = height;
                
                // Desenhar imagem redimensionada
                ctx.drawImage(img, 0, 0, width, height);
                
                // Converter para blob com compressão
                canvas.toBlob((blob) => {
                    if (blob) {
                        // Criar arquivo com o mesmo nome
                        const compressedFile = new File([blob], file.name, {
                            type: 'image/jpeg',
                            lastModified: Date.now()
                        });
                        resolve(compressedFile);
                    } else {
                        reject(new Error('Erro ao comprimir imagem'));
                    }
                }, 'image/jpeg', 0.85); // 85% qualidade
            };
            
            img.onerror = () => reject(new Error('Erro ao carregar imagem'));
            img.src = e.target.result;
        };
        
        reader.onerror = () => reject(new Error('Erro ao ler arquivo'));
        reader.readAsDataURL(file);
    });
}

// Comprimir vídeo usando Canvas + MediaRecorder
async function compressVideo(file, onProgress) {
    return new Promise((resolve, reject) => {
        const video = document.createElement('video');
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');
        
        video.preload = 'metadata';
        video.muted = true; // Silenciar preview, mas vamos capturar áudio separado
        
        video.onloadedmetadata = async () => {
            // Definir dimensões (máximo 1280px de largura)
            let width = video.videoWidth;
            let height = video.videoHeight;
            const maxWidth = 1280;
            
            if (width > maxWidth) {
                height = (height / width) * maxWidth;
                width = maxWidth;
            }
            
            canvas.width = width;
            canvas.height = height;
            
            // Capturar stream de vídeo do canvas
            const videoStream = canvas.captureStream(30); // 30 FPS
            
            // Tentar capturar áudio do vídeo original
            let finalStream = videoStream;
            
            try {
                // Criar elemento de vídeo separado para áudio
                const audioVideo = document.createElement('video');
                audioVideo.src = URL.createObjectURL(file);
                audioVideo.muted = false;
                
                await audioVideo.play();
                
                // Capturar stream de áudio
                const audioContext = new AudioContext();
                const source = audioContext.createMediaElementSource(audioVideo);
                const destination = audioContext.createMediaStreamDestination();
                source.connect(destination);
                source.connect(audioContext.destination);
                
                // Combinar vídeo (canvas) + áudio (original)
                const audioTrack = destination.stream.getAudioTracks()[0];
                if (audioTrack) {
                    finalStream.addTrack(audioTrack);
                }
                
                // Pausar e sincronizar com vídeo principal
                audioVideo.pause();
                audioVideo.currentTime = 0;
                
            } catch (error) {
                console.warn('Não foi possível capturar áudio:', error);
                // Continua sem áudio
            }
            
            const chunks = [];
            
            // Criar MediaRecorder com compressão
            const options = {
                mimeType: 'video/webm;codecs=vp9,opus', // VP9 vídeo + Opus áudio
                videoBitsPerSecond: 1000000 // 1 Mbps
            };
            
            // Fallback para VP8 se VP9 não disponível
            if (!MediaRecorder.isTypeSupported(options.mimeType)) {
                options.mimeType = 'video/webm;codecs=vp8,opus';
            }
            
            // Fallback sem codec específico
            if (!MediaRecorder.isTypeSupported(options.mimeType)) {
                options.mimeType = 'video/webm';
            }
            
            const mediaRecorder = new MediaRecorder(finalStream, options);
            
            mediaRecorder.ondataavailable = (e) => {
                if (e.data.size > 0) {
                    chunks.push(e.data);
                }
            };
            
            mediaRecorder.onstop = () => {
                const blob = new Blob(chunks, { type: options.mimeType });
                const compressedFile = new File([blob], file.name.replace(/\.[^/.]+$/, '.webm'), {
                    type: options.mimeType,
                    lastModified: Date.now()
                });
                resolve(compressedFile);
            };
            
            // Iniciar gravação
            mediaRecorder.start();
            video.play();
            
            const duration = video.duration;
            
            // Desenhar frames no canvas
            const drawFrame = () => {
                if (video.paused || video.ended) {
                    mediaRecorder.stop();
                    return;
                }
                
                ctx.drawImage(video, 0, 0, width, height);
                
                // Atualizar progresso
                const currentTime = video.currentTime;
                const percent = Math.round((currentTime / duration) * 100);
                onProgress(percent);
                
                requestAnimationFrame(drawFrame);
            };
            
            video.onended = () => {
                mediaRecorder.stop();
            };
            
            drawFrame();
        };
        
        video.onerror = () => reject(new Error('Erro ao carregar vídeo'));
        video.src = URL.createObjectURL(file);
    });
}

// Upload com progresso usando XMLHttpRequest
function uploadWithProgress(formData, onProgress) {
    return new Promise((resolve, reject) => {
        const xhr = new XMLHttpRequest();
        
        // Evento de progresso
        xhr.upload.addEventListener('progress', (e) => {
            if (e.lengthComputable) {
                const percent = Math.round((e.loaded / e.total) * 100);
                onProgress(percent);
            }
        });
        
        // Quando completar
        xhr.addEventListener('load', () => {
            if (xhr.status === 200) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    resolve(response);
                } catch (error) {
                    reject(new Error('Erro ao processar resposta'));
                }
            } else {
                reject(new Error('Erro no upload'));
            }
        });
        
        // Erro de rede
        xhr.addEventListener('error', () => {
            reject(new Error('Erro de conexão'));
        });
        
        // Enviar requisição
        xhr.open('POST', '/mensagem/backend/upload_media.php');
        xhr.send(formData);
    });
}

// ===== VIDEO PLAYER PERSONALIZADO =====
let currentVideo = null;

function openVideoPlayer(videoSrc) {
    const modal = document.getElementById('videoPlayerModal');
    const video = document.getElementById('fullscreenVideo');
    const playBtn = document.getElementById('videoPlayBtn');
    const playPauseBtn = document.getElementById('playPauseBtn');
    const progressSlider = document.getElementById('progressSlider');
    const progressFilled = document.getElementById('progressFilled');
    const timeDisplay = document.getElementById('timeDisplay');
    const volumeSlider = document.getElementById('volumeSlider');
    const volumeBtn = document.getElementById('volumeBtn');
    
    currentVideo = video;
    video.src = videoSrc;
    modal.style.display = 'flex';
    
    // Pausar vídeo ao abrir
    video.pause();
    playBtn.innerHTML = '<i class="fas fa-play"></i>';
    playPauseBtn.innerHTML = '<i class="fas fa-play"></i>';
    
    // Atualizar progresso
    video.addEventListener('timeupdate', () => {
        const percent = (video.currentTime / video.duration) * 100;
        progressFilled.style.width = percent + '%';
        progressSlider.value = percent;
        
        const currentMin = Math.floor(video.currentTime / 60);
        const currentSec = Math.floor(video.currentTime % 60);
        const durationMin = Math.floor(video.duration / 60);
        const durationSec = Math.floor(video.duration % 60);
        
        timeDisplay.textContent = `${currentMin}:${currentSec.toString().padStart(2, '0')} / ${durationMin}:${durationSec.toString().padStart(2, '0')}`;
    });
    
    // Slider de progresso
    progressSlider.addEventListener('input', (e) => {
        const time = (e.target.value / 100) * video.duration;
        video.currentTime = time;
    });
    
    // Slider de volume
    volumeSlider.addEventListener('input', (e) => {
        video.volume = e.target.value / 100;
        updateVolumeIcon();
    });
    
    // Quando o vídeo terminar
    video.addEventListener('ended', () => {
        playBtn.classList.remove('playing');
        playBtn.innerHTML = '<i class="fas fa-play"></i>';
        playPauseBtn.innerHTML = '<i class="fas fa-play"></i>';
    });
}

// Abrir modal de visualização de imagem
function openImageViewer(imageSrc) {
    const modal = document.getElementById('imageViewerModal');
    const img = document.getElementById('fullscreenImage');
    
    img.src = imageSrc;
    modal.style.display = 'flex';
}

// Fechar modal de imagem
function closeImageViewer() {
    const modal = document.getElementById('imageViewerModal');
    const img = document.getElementById('fullscreenImage');
    
    img.src = '';
    modal.style.display = 'none';
}

function closeVideoPlayer() {
    const modal = document.getElementById('videoPlayerModal');
    const video = document.getElementById('fullscreenVideo');
    
    video.pause();
    video.src = '';
    modal.style.display = 'none';
    currentVideo = null;
}

function togglePlayPause() {
    const video = document.getElementById('fullscreenVideo');
    const playBtn = document.getElementById('videoPlayBtn');
    const playPauseBtn = document.getElementById('playPauseBtn');
    
    if (video.paused) {
        video.play();
        playBtn.classList.add('playing');
        playBtn.innerHTML = '<i class="fas fa-pause"></i>';
        playPauseBtn.innerHTML = '<i class="fas fa-pause"></i>';
    } else {
        video.pause();
        playBtn.classList.remove('playing');
        playBtn.innerHTML = '<i class="fas fa-play"></i>';
        playPauseBtn.innerHTML = '<i class="fas fa-play"></i>';
    }
}

function toggleMute() {
    const video = document.getElementById('fullscreenVideo');
    const volumeBtn = document.getElementById('volumeBtn');
    const volumeSlider = document.getElementById('volumeSlider');
    
    video.muted = !video.muted;
    
    if (video.muted) {
        volumeBtn.innerHTML = '<i class="fas fa-volume-mute"></i>';
        volumeSlider.value = 0;
    } else {
        volumeSlider.value = video.volume * 100;
        updateVolumeIcon();
    }
}

function updateVolumeIcon() {
    const video = document.getElementById('fullscreenVideo');
    const volumeBtn = document.getElementById('volumeBtn');
    
    if (video.muted || video.volume === 0) {
        volumeBtn.innerHTML = '<i class="fas fa-volume-mute"></i>';
    } else if (video.volume < 0.5) {
        volumeBtn.innerHTML = '<i class="fas fa-volume-down"></i>';
    } else {
        volumeBtn.innerHTML = '<i class="fas fa-volume-up"></i>';
    }
}

// ===== GRAVAÇÃO DE ÁUDIO =====

// Event Listeners para botão de voz
btnVoice.addEventListener('mousedown', startRecording);
btnVoice.addEventListener('touchstart', startRecording);
btnVoice.addEventListener('mouseup', stopRecording);
btnVoice.addEventListener('touchend', stopRecording);
btnVoice.addEventListener('mouseleave', cancelRecording);

async function startRecording(e) {
    e.preventDefault();
    
    if (isRecording) return;
    
    try {
        // Solicitar permissão de microfone
        audioStream = await navigator.mediaDevices.getUserMedia({ audio: true });
        
        // Configurar MediaRecorder
        const options = { mimeType: 'audio/webm;codecs=opus' };
        
        // Fallback para outros formatos
        if (!MediaRecorder.isTypeSupported(options.mimeType)) {
            options.mimeType = 'audio/webm';
        }
        if (!MediaRecorder.isTypeSupported(options.mimeType)) {
            options.mimeType = 'audio/ogg;codecs=opus';
        }
        
        mediaRecorder = new MediaRecorder(audioStream, options);
        audioChunks = [];
        
        mediaRecorder.ondataavailable = (event) => {
            if (event.data.size > 0) {
                audioChunks.push(event.data);
            }
        };
        
        mediaRecorder.onstop = async () => {
            const audioBlob = new Blob(audioChunks, { type: mediaRecorder.mimeType });
            await sendAudio(audioBlob);
        };
        
        // Iniciar gravação
        mediaRecorder.start();
        isRecording = true;
        
        // Mostrar indicador
        recordingIndicator.style.display = 'block';
        btnVoice.classList.add('recording');
        
        // Iniciar contador
        recordingStartTime = Date.now();
        updateRecordingTime();
        recordingTimer = setInterval(updateRecordingTime, 100);
        
    } catch (error) {
        console.error('Erro ao acessar microfone:', error);
        if (error.name === 'NotAllowedError') {
            showNotification('Permissão de microfone negada', 'error');
        } else {
            showNotification('Erro ao acessar microfone', 'error');
        }
    }
}

function stopRecording(e) {
    e.preventDefault();
    
    if (!isRecording) return;
    
    isRecording = false;
    
    // Parar gravação
    if (mediaRecorder && mediaRecorder.state !== 'inactive') {
        mediaRecorder.stop();
    }
    
    // Parar stream
    if (audioStream) {
        audioStream.getTracks().forEach(track => track.stop());
        audioStream = null;
    }
    
    // Esconder indicador
    recordingIndicator.style.display = 'none';
    btnVoice.classList.remove('recording');
    
    // Limpar timer
    if (recordingTimer) {
        clearInterval(recordingTimer);
        recordingTimer = null;
    }
}

function cancelRecording(e) {
    if (!isRecording) return;
    
    isRecording = false;
    
    // Parar sem enviar
    if (mediaRecorder && mediaRecorder.state !== 'inactive') {
        mediaRecorder.ondataavailable = null; // Evitar envio
        mediaRecorder.stop();
    }
    
    // Parar stream
    if (audioStream) {
        audioStream.getTracks().forEach(track => track.stop());
        audioStream = null;
    }
    
    // Esconder indicador
    recordingIndicator.style.display = 'none';
    btnVoice.classList.remove('recording');
    
    // Limpar timer
    if (recordingTimer) {
        clearInterval(recordingTimer);
        recordingTimer = null;
    }
    
    audioChunks = [];
}

function updateRecordingTime() {
    if (!recordingStartTime) return;
    
    const elapsed = Date.now() - recordingStartTime;
    const seconds = Math.floor(elapsed / 1000);
    const minutes = Math.floor(seconds / 60);
    const secs = seconds % 60;
    
    recordingTime.textContent = `${minutes}:${secs.toString().padStart(2, '0')}`;
    
    // Limite de 2 minutos
    if (seconds >= 120) {
        stopRecording(new Event('click'));
        showNotification('Limite de 2 minutos atingido', 'info');
    }
}

async function sendAudio(audioBlob) {
    try {
        // Criar arquivo
        const audioFile = new File([audioBlob], `audio_${Date.now()}.webm`, {
            type: audioBlob.type,
            lastModified: Date.now()
        });
        
        // Upload do áudio
        const formData = new FormData();
        formData.append('audio', audioFile);
        
        const uploadResponse = await fetch('/mensagem/backend/upload_audio.php', {
            method: 'POST',
            body: formData
        });
        
        const uploadData = await uploadResponse.json();
        
        if (!uploadData.success) {
            showNotification(uploadData.message || 'Erro ao enviar áudio', 'error');
            return;
        }
        
        // Enviar mensagem com o áudio
        const messageFormData = new FormData();
        messageFormData.append('destinatario_id', otherUserId);
        messageFormData.append('mensagem', '');
        messageFormData.append('tipo_midia', 'audio');
        messageFormData.append('caminho_midia', uploadData.path);
        
        const messageResponse = await fetch('/mensagem/backend/send_message.php', {
            method: 'POST',
            body: messageFormData
        });
        
        const messageData = await messageResponse.json();
        
        if (messageData.success) {
            // Atualizar saldo e mostrar efeito apenas se pode ganhar
            if (messageData.saldo && messageData.ganhou) {
                saldoValue.textContent = `R$ ${messageData.saldo}`;
                showMoneyToast();
                startCooldown();
            } else if (messageData.saldo) {
                saldoValue.textContent = `R$ ${messageData.saldo}`;
            }
            
            await loadMessages(true);
        } else {
            showNotification(messageData.message || 'Erro ao enviar mensagem', 'error');
        }
        
    } catch (error) {
        console.error('Erro ao enviar áudio:', error);
        showNotification('Erro ao enviar áudio', 'error');
    }
}

// ===== SISTEMA DE SALDO =====

async function loadSaldo() {
    try {
        const response = await fetch('/mensagem/backend/get_saldo.php');
        const data = await response.json();
        
        if (data.success) {
            saldoValue.textContent = `R$ ${data.saldo}`;
            
            // Verificar se deve bloquear o usuário
            const saldoNumerico = parseFloat(data.saldo.replace(',', '.'));
            const isMember = data.is_member || false;
            
            if (saldoNumerico >= 50 && !isMember) {
                userBlocked = true;
            } else {
                userBlocked = false;
            }
        }
    } catch (error) {
        console.error('Erro ao carregar saldo:', error);
    }
}

function showMoneyToast(valor = '1,00') {
    // Atualizar texto do toast com o valor recebido
    moneyToast.querySelector('span').textContent = `+R$ ${valor}`;
    
    // Tocar som de dinheiro
    moneySound.currentTime = 0;
    moneySound.play().catch(err => console.log('Erro ao tocar som:', err));
    
    // Mostrar toast
    moneyToast.classList.add('show');
    
    // Animar saldo
    saldoDisplay.classList.add('animate');
    setTimeout(() => {
        saldoDisplay.classList.remove('animate');
    }, 600);
    
    // Esconder toast após 3 segundos
    setTimeout(() => {
        moneyToast.classList.remove('show');
    }, 3000);
}

function startCooldown() {
    // Bloquear ganho
    canEarnMoney = false;
    
    // Iniciar animação da barra
    cooldownBar.classList.remove('active');
    void cooldownBar.offsetWidth; // Force reflow
    cooldownBar.classList.add('active');
    
    // Limpar timer anterior se existir
    if (cooldownTimer) {
        clearTimeout(cooldownTimer);
    }
    
    // Após 10 segundos, permitir ganhar novamente
    cooldownTimer = setTimeout(() => {
        canEarnMoney = true;
        cooldownBar.classList.remove('active');
    }, 10000);
}

// ===== MENU E BLOQUEIO =====

// Toggle menu
btnMenu.addEventListener('click', (e) => {
    e.stopPropagation();
    menuDropdown.classList.toggle('show');
});

// Fechar menu ao clicar fora
document.addEventListener('click', () => {
    menuDropdown.classList.remove('show');
});

// Bloquear/Desbloquear usuário
let isUserBlocked = false;

// Verificar status de bloqueio ao carregar
async function checkBlockStatus() {
    if (!otherUserId) return;
    
    try {
        const formData = new FormData();
        formData.append('other_user_id', otherUserId);
        
        const response = await fetch('/mensagem/backend/check_block_status.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            isUserBlocked = data.is_blocked;
            updateBlockButtonUI();
        }
    } catch (error) {
        console.error('Erro ao verificar bloqueio:', error);
    }
}

function updateBlockButtonUI() {
    if (isUserBlocked) {
        btnBlock.innerHTML = '<i class="fas fa-unlock"></i><span>Desbloquear Usuário</span>';
        btnBlock.style.color = '#10b981';
    } else {
        btnBlock.innerHTML = '<i class="fas fa-ban"></i><span>Bloquear Usuário</span>';
        btnBlock.style.color = '';
    }
    btnBlock.disabled = false;
}

btnBlock.addEventListener('click', async () => {
    if (!otherUserId) return;
    
    if (isUserBlocked) {
        // Desbloquear
        const confirmar = confirm(`Deseja desbloquear ${otherUserData.nome}? Você voltará a receber mensagens dessa pessoa.`);
        
        if (confirmar) {
            try {
                const formData = new FormData();
                formData.append('blocked_user_id', otherUserId);
                
                const response = await fetch('/mensagem/backend/unblock_user.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showNotification('Usuário desbloqueado com sucesso', 'success');
                    menuDropdown.classList.remove('show');
                    isUserBlocked = false;
                    updateBlockButtonUI();
                } else {
                    showNotification(data.message || 'Erro ao desbloquear usuário', 'error');
                }
            } catch (error) {
                console.error('Erro:', error);
                showNotification('Erro ao desbloquear usuário', 'error');
            }
        }
    } else {
        // Bloquear
        const confirmar = confirm(`Deseja realmente bloquear ${otherUserData.nome}? Você não receberá mais mensagens dessa pessoa.`);
        
        if (confirmar) {
            try {
                const formData = new FormData();
                formData.append('blocked_user_id', otherUserId);
                
                const response = await fetch('/mensagem/backend/block_user.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showNotification('Usuário bloqueado com sucesso', 'success');
                    menuDropdown.classList.remove('show');
                    isUserBlocked = true;
                    updateBlockButtonUI();
                } else {
                    showNotification(data.message || 'Erro ao bloquear usuário', 'error');
                }
            } catch (error) {
                console.error('Erro:', error);
                showNotification('Erro ao bloquear usuário', 'error');
            }
        }
    }
});

// ===== SISTEMA DE RESPOSTA (SWIPE) =====

let swipeState = {
    startX: 0,
    currentX: 0,
    isDragging: false,
    messageElement: null
};

// Fechar preview de resposta
replyPreviewClose.addEventListener('click', () => {
    replyPreview.style.display = 'none';
    replyingTo = null;
});

// Adicionar swipe às mensagens
function addSwipeToMessage(messageElement, messageData) {
    const bubble = messageElement.querySelector('.message-bubble');
    const isMine = messageElement.classList.contains('mine');
    
    // Adicionar ícone de resposta
    const replyIcon = document.createElement('div');
    replyIcon.className = 'reply-icon';
    replyIcon.innerHTML = '<i class="fas fa-reply"></i>';
    messageElement.appendChild(replyIcon);
    
    let startX = 0;
    let startY = 0;
    let currentX = 0;
    let isDragging = false;
    let hasMoved = false;
    
    const onStart = (e) => {
        const touch = e.touches ? e.touches[0] : e;
        startX = touch.clientX;
        startY = touch.clientY;
        currentX = startX;
        isDragging = true;
        hasMoved = false;
    };
    
    const onMove = (e) => {
        if (!isDragging) return;
        
        const touch = e.touches ? e.touches[0] : e;
        currentX = touch.clientX;
        const currentY = touch.clientY;
        const diffX = currentX - startX;
        const diffY = currentY - startY;
        
        // Verificar se realmente moveu (threshold de 5px)
        if (Math.abs(diffX) > 5 || Math.abs(diffY) > 5) {
            hasMoved = true;
        }
        
        if (!hasMoved) return;
        
        messageElement.classList.add('swiping');
        
        // Limitar swipe
        const maxSwipe = 80;
        let swipeAmount = 0;
        
        if (isMine) {
            // Minhas mensagens: swipe para esquerda
            swipeAmount = Math.max(-maxSwipe, Math.min(0, diffX));
        } else {
            // Mensagens recebidas: swipe para direita
            swipeAmount = Math.min(maxSwipe, Math.max(0, diffX));
        }
        
        messageElement.style.transform = `translateX(${swipeAmount}px)`;
        
        // Mostrar ícone quando swipe > 40px
        if (Math.abs(swipeAmount) > 40) {
            messageElement.classList.add('show-reply-icon');
        } else {
            messageElement.classList.remove('show-reply-icon');
        }
    };
    
    const onEnd = () => {
        if (!isDragging) return;
        
        isDragging = false;
        messageElement.classList.remove('swiping');
        
        const diff = currentX - startX;
        
        // Se swipe > 60px E realmente moveu, ativar resposta
        if (hasMoved && Math.abs(diff) > 60) {
            setReplyTo(messageData);
        }
        
        // Resetar posição
        messageElement.style.transform = '';
        messageElement.classList.remove('show-reply-icon');
        hasMoved = false;
    };
    
    // Touch events
    messageElement.addEventListener('touchstart', onStart, { passive: true });
    messageElement.addEventListener('touchmove', onMove, { passive: true });
    messageElement.addEventListener('touchend', onEnd);
    
    // Mouse events (para desktop)
    messageElement.addEventListener('mousedown', onStart);
    messageElement.addEventListener('mousemove', onMove);
    messageElement.addEventListener('mouseup', onEnd);
    messageElement.addEventListener('mouseleave', onEnd);
}

// Mostrar opção de deletar mensagem
function showDeleteOption(messageElement, messageData) {
    // Verificar se já existe um modal de delete aberto
    const existingModal = document.querySelector('.delete-message-modal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // Criar modal de confirmação
    const modal = document.createElement('div');
    modal.className = 'delete-message-modal';
    modal.innerHTML = `
        <div class="delete-modal-overlay"></div>
        <div class="delete-modal-content">
            <h3>Apagar Mensagem?</h3>
            <p>Esta mensagem será apagada para todos.</p>
            <div class="delete-modal-buttons">
                <button class="btn-cancel-delete">Cancelar</button>
                <button class="btn-confirm-delete">Apagar</button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Event listeners
    const btnCancel = modal.querySelector('.btn-cancel-delete');
    const btnConfirm = modal.querySelector('.btn-confirm-delete');
    const overlay = modal.querySelector('.delete-modal-overlay');
    
    const closeModal = () => {
        modal.remove();
    };
    
    btnCancel.addEventListener('click', closeModal);
    overlay.addEventListener('click', closeModal);
    
    btnConfirm.addEventListener('click', async () => {
        closeModal();
        await deleteMessage(messageData.id, messageElement);
    });
    
    // Animar entrada
    setTimeout(() => {
        modal.classList.add('show');
    }, 10);
}

// Deletar mensagem
async function deleteMessage(messageId, messageElement) {
    try {
        const response = await fetch('/mensagem/backend/delete_message.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ message_id: messageId })
        });
        
        const data = await response.json();
        
        if (data.success) {
            // Animar remoção
            messageElement.style.opacity = '0';
            messageElement.style.transform = 'scale(0.8)';
            messageElement.style.transition = 'all 0.3s ease';
            
            setTimeout(() => {
                messageElement.remove();
            }, 300);
        } else {
            showNotification(data.message || 'Erro ao deletar mensagem', 'error');
        }
    } catch (error) {
        console.error('Erro ao deletar:', error);
        showNotification('Erro ao deletar mensagem', 'error');
    }
}

// Definir mensagem para responder
function setReplyTo(messageData) {
    replyingTo = messageData;
    
    // Atualizar preview
    replyPreviewName.textContent = messageData.is_mine ? 'Você' : otherUserData.nome.split(' ')[0];
    
    let previewText = '';
    if (messageData.tipo_midia === 'gift' || messageData.tipo === 'gift') {
        const valor = parseFloat(messageData.valor || messageData.gift_amount || 0).toFixed(2).replace('.', ',');
        previewText = `🎁 Presente R$ ${valor}`;
    } else if (messageData.tipo_midia === 'image') {
        previewText = '📷 Imagem';
    } else if (messageData.tipo_midia === 'video') {
        previewText = '🎥 Vídeo';
    } else if (messageData.tipo_midia === 'audio') {
        previewText = '🎤 Áudio';
    } else {
        previewText = messageData.mensagem || '';
    }
    
    replyPreviewText.textContent = previewText;
    replyPreview.style.display = 'flex';
    
    // Focar no input
    messageInput.focus();
}

// Mostrar atalhos de mensagem rápida
function showQuickReplies() {
    if (!hasInteraction && quickReplies) {
        quickReplies.style.display = 'flex';
    }
}

// Esconder atalhos de mensagem rápida
function hideQuickReplies() {
    if (quickReplies) {
        quickReplies.style.display = 'none';
    }
}

// Event listeners dos atalhos de mensagem rápida
if (quickReplies) {
    const quickReplyButtons = quickReplies.querySelectorAll('.quick-reply-btn');
    quickReplyButtons.forEach(btn => {
        btn.addEventListener('click', async () => {
            const message = btn.getAttribute('data-message');
            if (message) {
                // Preencher input e enviar
                messageInput.value = message;
                await sendMessage();
                // Esconder atalhos após envio
                hasInteraction = true;
                hideQuickReplies();
            }
        });
    });
}

// Event listeners do modal de membership
btnUpgrade.addEventListener('click', () => {
    window.location.href = CONFIG.payment.membership;
});

btnCancelMembership.addEventListener('click', () => {
    membershipModal.style.display = 'none';
});

// Verificar se volta da página de liberação (ao ganhar foco)
window.addEventListener('focus', async () => {
    // Recarregar saldo para verificar se se tornou membro
    await loadSaldo();
});

// Função para mostrar modal de membership
function showMembershipModal() {
    membershipModal.style.display = 'flex';
}

// ===== SISTEMA DE PRESENTES =====

// Abrir modal de presente
function openGiftModal() {
    if (!otherUserId) {
        showNotification('Erro ao abrir modal de presente', 'error');
        return;
    }
    
    // Atualizar saldo no modal
    giftCurrentBalance.textContent = saldoValue.textContent;
    currentBalance = parseFloat(saldoValue.textContent.replace('R$', '').replace(',', '.').trim());
    
    // Resetar campos
    currentGiftAmount = 0;
    giftCustomAmount.value = '';
    giftMessage.value = '';
    giftCharCount.textContent = '0';
    
    // Remover seleção de botões
    document.querySelectorAll('.gift-amount-btn').forEach(btn => {
        btn.classList.remove('selected');
    });
    
    giftModal.style.display = 'flex';
}

// Fechar modal de presente
function closeGiftModal() {
    giftModal.style.display = 'none';
}

// Selecionar valor pré-definido
document.querySelectorAll('.gift-amount-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        const amount = parseFloat(this.getAttribute('data-amount'));
        
        // Verificar se tem saldo suficiente
        if (amount > currentBalance) {
            showNotification('Saldo insuficiente para este valor', 'error');
            return;
        }
        
        // Remover seleção anterior
        document.querySelectorAll('.gift-amount-btn').forEach(b => b.classList.remove('selected'));
        
        // Selecionar este botão
        this.classList.add('selected');
        currentGiftAmount = amount;
        
        // Limpar input customizado
        giftCustomAmount.value = '';
    });
});

// Input customizado de valor
giftCustomAmount.addEventListener('input', function() {
    const value = parseFloat(this.value) || 0;
    
    if (value > currentBalance) {
        showNotification('Saldo insuficiente', 'error');
        this.value = currentBalance.toFixed(2);
        currentGiftAmount = currentBalance;
    } else if (value < 0) {
        this.value = '';
        currentGiftAmount = 0;
    } else {
        currentGiftAmount = value;
    }
    
    // Remover seleção de botões pré-definidos
    document.querySelectorAll('.gift-amount-btn').forEach(b => b.classList.remove('selected'));
});

// Contador de caracteres da mensagem
giftMessage.addEventListener('input', function() {
    giftCharCount.textContent = this.value.length;
});

// Enviar presente
btnSendGift.addEventListener('click', async function() {
    if (currentGiftAmount <= 0) {
        showNotification('Selecione um valor para enviar', 'error');
        return;
    }
    
    if (currentGiftAmount > currentBalance) {
        showNotification('Saldo insuficiente', 'error');
        return;
    }
    
    if (!otherUserId) {
        showNotification('Erro ao identificar destinatário', 'error');
        return;
    }
    
    // Desabilitar botão
    btnSendGift.disabled = true;
    btnSendGift.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Enviando...';
    
    try {
        const response = await fetch('/mensagem/backend/send_gift.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                destinatario_id: otherUserId,
                valor: currentGiftAmount,
                mensagem: giftMessage.value.trim()
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            // Fechar modal
            closeGiftModal();
            
            // Atualizar saldo
            saldoValue.textContent = `R$ ${data.novo_saldo}`;
            currentBalance = parseFloat(data.novo_saldo);
            
            // Adicionar mensagem de presente no chat
            const giftMessageData = {
                id: data.mensagem_id,
                tipo: 'gift',
                valor: currentGiftAmount,
                mensagem: giftMessage.value.trim(),
                is_mine: true,
                time: new Date().toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' }),
                lida: false
            };
            
            addGiftMessageToUI(giftMessageData);
            lastMessageId = Math.max(lastMessageId, data.mensagem_id);
            
            showNotification('🎁 Presente enviado com sucesso!', 'success');
            
            // Tocar som
            moneySound.play();
        } else {
            showNotification(data.message || 'Erro ao enviar presente', 'error');
        }
    } catch (error) {
        console.error('Erro ao enviar presente:', error);
        showNotification('Erro ao enviar presente', 'error');
    } finally {
        // Reabilitar botão
        btnSendGift.disabled = false;
        btnSendGift.innerHTML = '<i class="fas fa-gift"></i> Enviar Presente';
    }
});

// Adicionar mensagem de presente na UI
function addGiftMessageToUI(giftData) {
    const messageDiv = document.createElement('div');
    messageDiv.className = `message ${giftData.is_mine ? 'mine' : 'other'}`;
    messageDiv.setAttribute('data-message-id', giftData.id);
    
    const bubbleDiv = document.createElement('div');
    bubbleDiv.className = 'message-bubble';
    
    const giftDiv = document.createElement('div');
    giftDiv.className = 'message-gift';
    giftDiv.innerHTML = `
        <div class="gift-bubble-icon">🎁</div>
        <div class="gift-bubble-details">
            <div class="gift-bubble-amount">R$ ${giftData.valor.toFixed(2).replace('.', ',')}</div>
            <div class="gift-bubble-label">Presente ${giftData.is_mine ? 'enviado' : 'recebido'}</div>
            ${giftData.mensagem ? `<div class="gift-bubble-message">${giftData.mensagem}</div>` : ''}
        </div>
    `;
    
    bubbleDiv.appendChild(giftDiv);
    
    const timeDiv = document.createElement('div');
    timeDiv.className = 'message-time';
    timeDiv.innerHTML = `<span>${giftData.time}</span>`;
    if (giftData.is_mine) {
        timeDiv.innerHTML += `<i class="fas fa-check status-icon ${giftData.lida ? 'read' : ''}"></i>`;
    }
    
    bubbleDiv.appendChild(timeDiv);
    messageDiv.appendChild(bubbleDiv);
    messagesContainer.appendChild(messageDiv);
    
    scrollToBottom();
}

// Event listener do botão de presente no menu
if (btnGiftMenu) {
    btnGiftMenu.addEventListener('click', () => {
        openGiftModal();
        menuDropdown.classList.remove('show'); // Fechar menu ao clicar
    });
}

// Inicializar
initChat();
