<?php
/**
 * Verificar se mensagem contém palavras proibidas
 */

require_once 'config.php';

function verificarPalavrasProibidas($mensagem, $user_id, $destinatario_id) {
    try {
        $conn = getConnection();
        
        // Buscar todas as palavras proibidas
        $stmt = $conn->query("SELECT palavra, tipo FROM palavras_proibidas");
        $palavras = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Normalizar mensagem atual: remover espaços, hífens, underscores, pontos
        $mensagemNormalizada = mb_strtolower($mensagem, 'UTF-8');
        $mensagemNormalizada = preg_replace('/[\s\-_\.]+/', '', $mensagemNormalizada);
        
        // Buscar últimas 3 mensagens do mesmo usuário para o mesmo destinatário
        $stmt = $conn->prepare("
            SELECT mensagem 
            FROM mensagens 
            WHERE remetente_id = ? 
            AND destinatario_id = ? 
            AND created_at > NOW() - INTERVAL 1 MINUTE
            ORDER BY created_at DESC 
            LIMIT 3
        ");
        $stmt->execute([$user_id, $destinatario_id]);
        $mensagensRecentes = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        // Concatenar últimas mensagens + mensagem atual
        $mensagemCompleta = '';
        foreach (array_reverse($mensagensRecentes) as $msgAntiga) {
            $msgNormalizada = mb_strtolower($msgAntiga, 'UTF-8');
            $msgNormalizada = preg_replace('/[\s\-_\.]+/', '', $msgNormalizada);
            $mensagemCompleta .= $msgNormalizada;
        }
        $mensagemCompleta .= $mensagemNormalizada;
        
        // Verificar cada palavra proibida na mensagem atual E na combinação
        foreach ($palavras as $palavra) {
            $palavraNormalizada = mb_strtolower($palavra['palavra'], 'UTF-8');
            $palavraNormalizada = preg_replace('/[\s\-_\.]+/', '', $palavraNormalizada);
            
            if ($palavra['tipo'] === 'exata') {
                // Palavra exata (mensagem completa)
                if ($mensagemNormalizada === $palavraNormalizada || $mensagemCompleta === $palavraNormalizada) {
                    return registrarAviso($user_id, $palavra['palavra']);
                }
            } else {
                // Palavra contida (busca na mensagem atual OU na combinação)
                if (strpos($mensagemNormalizada, $palavraNormalizada) !== false || 
                    strpos($mensagemCompleta, $palavraNormalizada) !== false) {
                    return registrarAviso($user_id, $palavra['palavra']);
                }
            }
        }
        
        return ['permitido' => true];
        
    } catch (PDOException $e) {
        error_log("Erro ao verificar palavras: " . $e->getMessage());
        return ['permitido' => true]; // Em caso de erro, permitir envio
    }
}

function registrarAviso($user_id, $palavra_detectada) {
    try {
        $conn = getConnection();
        
        // Incrementar contador de avisos
        $stmt = $conn->prepare("
            UPDATE usuarios 
            SET avisos_palavras = avisos_palavras + 1 
            WHERE id = ?
        ");
        $stmt->execute([$user_id]);
        
        // Buscar total de avisos
        $stmt = $conn->prepare("SELECT avisos_palavras FROM usuarios WHERE id = ?");
        $stmt->execute([$user_id]);
        $avisos = $stmt->fetchColumn();
        
        // Se atingiu 5 avisos, excluir completamente o usuário
        if ($avisos >= 5) {
            // Buscar foto de perfil do usuário
            $stmt = $conn->prepare("SELECT foto_perfil FROM usuarios WHERE id = ?");
            $stmt->execute([$user_id]);
            $foto_perfil = $stmt->fetchColumn();

            // 1. Buscar todas as conversas do usuário
            $stmt = $conn->prepare("SELECT id FROM conversas WHERE user1_id = ? OR user2_id = ?");
            $stmt->execute([$user_id, $user_id]);
            $conversas = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // 2. Buscar e deletar todos os arquivos de mídia das mensagens
            foreach ($conversas as $conversa) {
                $stmt = $conn->prepare("SELECT caminho_midia FROM mensagens WHERE conversa_id = ? AND caminho_midia IS NOT NULL AND caminho_midia != ''");
                $stmt->execute([$conversa['id']]);
                $midias = $stmt->fetchAll(PDO::FETCH_ASSOC);
                foreach ($midias as $midia) {
                    $filePath = dirname(__DIR__) . "/" . $midia['caminho_midia'];
                    if (file_exists($filePath)) {
                        unlink($filePath);
                    }
                }
                // Deletar mensagens
                $stmt = $conn->prepare("DELETE FROM mensagens WHERE conversa_id = ?");
                $stmt->execute([$conversa['id']]);
            }

            // 3. Excluir todas as conversas
            $stmt = $conn->prepare("DELETE FROM conversas WHERE user1_id = ? OR user2_id = ?");
            $stmt->execute([$user_id, $user_id]);

            // 4. Excluir saques se existir tabela
            $stmt = $conn->query("SHOW TABLES LIKE 'saques'");
            if ($stmt->rowCount() > 0) {
                $stmt = $conn->prepare("DELETE FROM saques WHERE user_id = ?");
                $stmt->execute([$user_id]);
            }

            // 5. Excluir foto de perfil do servidor
            if ($foto_perfil && file_exists(dirname(__DIR__) . "/" . $foto_perfil)) {
                unlink(dirname(__DIR__) . "/" . $foto_perfil);
            }

            // 6. Excluir usuário
            $stmt = $conn->prepare("DELETE FROM usuarios WHERE id = ?");
            $stmt->execute([$user_id]);

            // 7. Invalidar sessão do usuário
            if (session_status() === PHP_SESSION_ACTIVE) {
                session_destroy();
            }

            return [
                'permitido' => false,
                'banido' => true,
                'excluido' => true,
                'avisos' => $avisos,
                'palavra' => $palavra_detectada,
                'mensagem' => 'Sua conta foi excluída por violar as regras de uso repetidamente.'
            ];
        }
        
        return [
            'permitido' => false,
            'banido' => false,
            'avisos' => $avisos,
            'palavra' => $palavra_detectada,
            'mensagem' => "⚠️ Mensagem bloqueada: palavra proibida detectada. Aviso {$avisos}/5"
        ];
        
    } catch (PDOException $e) {
        error_log("Erro ao registrar aviso: " . $e->getMessage());
        return ['permitido' => true];
    }
}
?>
