<?php
/**
 * API para upload de mídia (imagens e vídeos)
 */

session_start();

require_once 'config.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Não autenticado']);
    exit;
}

// Validações
if (!isset($_FILES['media']) || !isset($_POST['type'])) {
    echo json_encode(['success' => false, 'message' => 'Arquivo ou tipo não fornecido']);
    exit;
}

$file = $_FILES['media'];
$type = $_POST['type']; // 'image' ou 'video'
$error = $file['error'];

// Verificar erro no upload
if ($error !== UPLOAD_ERR_OK) {
    echo json_encode(['success' => false, 'message' => 'Erro ao fazer upload do arquivo']);
    exit;
}

$fileSize = $file['size'];
$fileTmpName = $file['tmp_name'];
$fileName = $file['name'];
$fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

// Validar tipo de arquivo
if ($type === 'image') {
    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
    $maxSize = 3 * 1024 * 1024; // 3MB
    
    if (!in_array($fileExtension, $allowedExtensions)) {
        echo json_encode(['success' => false, 'message' => 'Formato de imagem não permitido. Use: JPG, PNG, GIF ou WebP']);
        exit;
    }
    
    if ($fileSize > $maxSize) {
        echo json_encode(['success' => false, 'message' => 'Imagem muito grande. Máximo 3MB']);
        exit;
    }
    
    // Validar se é realmente uma imagem
    $imageInfo = getimagesize($fileTmpName);
    if (!$imageInfo) {
        echo json_encode(['success' => false, 'message' => 'Arquivo não é uma imagem válida']);
        exit;
    }
    
} elseif ($type === 'video') {
    $allowedExtensions = ['mp4', 'webm', 'mov', 'avi'];
    
    if (!in_array($fileExtension, $allowedExtensions)) {
        echo json_encode(['success' => false, 'message' => 'Formato de vídeo não permitido. Use: MP4, WebM, MOV ou AVI']);
        exit;
    }
    
    // Validar tamanho do vídeo
    // Limite aumentado para aceitar vídeos curtos com boa qualidade
    $maxSize = 50 * 1024 * 1024; // 50MB (suficiente para ~30s em boa qualidade)
    
    if ($fileSize > $maxSize) {
        echo json_encode(['success' => false, 'message' => 'Vídeo muito grande. Máximo 50MB']);
        exit;
    }
    
    // Verificar duração usando ffprobe se disponível (opcional)
    if (function_exists('shell_exec')) {
        $ffprobe = @shell_exec("ffprobe -v error -show_entries format=duration -of default=noprint_wrappers=1:nokey=1 " . escapeshellarg($fileTmpName) . " 2>&1");
        
        if ($ffprobe !== null && is_numeric($ffprobe)) {
            $duration = floatval($ffprobe);
            if ($duration > 60) {
                echo json_encode(['success' => false, 'message' => 'Vídeo muito longo. Máximo 1 minuto']);
                exit;
            }
        }
    }
    
} else {
    echo json_encode(['success' => false, 'message' => 'Tipo de mídia inválido']);
    exit;
}

// Gerar nome único para o arquivo
$uniqueName = uniqid() . '_' . time() . '.' . $fileExtension;
$uploadDir = '../uploads/chat/';

// Criar diretório se não existir
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

$uploadPath = $uploadDir . $uniqueName;
$relativePath = 'uploads/chat/' . $uniqueName;

// Mover arquivo para diretório final
if (move_uploaded_file($fileTmpName, $uploadPath)) {
    echo json_encode([
        'success' => true,
        'path' => $relativePath,
        'type' => $type,
        'filename' => basename($uploadPath)
    ]);
} else {
    echo json_encode(['success' => false, 'message' => 'Erro ao salvar arquivo']);
}
?>
