<?php
/**
 * API para atualizar perfil do usuário
 * Salva foto de perfil, idade e sexo
 */

require_once 'config_global.php';
initSession();

// Verificar se o usuário está logado
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Usuário não autenticado'
    ]);
    exit;
}

// Verifica se é POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
    exit;
}

$userId = $_SESSION['user_id'];
$nome = isset($_POST['nome']) ? trim($_POST['nome']) : null;
$idade = isset($_POST['idade']) ? intval($_POST['idade']) : null;
$sexo = isset($_POST['sexo']) ? $_POST['sexo'] : null;

// Validações básicas
$updateFields = [];
$updateValues = [];

// Se está tentando atualizar nome
if ($nome !== null) {
    if (strlen($nome) < 3) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'O nome deve ter pelo menos 3 caracteres'
        ]);
        exit;
    }
    $updateFields[] = 'nome = ?';
    $updateValues[] = $nome;
}

// Se está tentando atualizar idade
if ($idade !== null) {
    if ($idade < 16 || $idade > 80) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'A idade deve estar entre 16 e 80 anos'
        ]);
        exit;
    }
    $updateFields[] = 'idade = ?';
    $updateValues[] = $idade;
}

// Se está tentando atualizar sexo
if ($sexo !== null) {
    if (!in_array($sexo, ['masculino', 'feminino', 'outro'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Sexo inválido'
        ]);
        exit;
    }
    $updateFields[] = 'sexo = ?';
    $updateValues[] = $sexo;
}

// Processar upload da foto
$fotoPath = null;

if (isset($_FILES['foto_perfil']) && $_FILES['foto_perfil']['error'] === UPLOAD_ERR_OK) {
    $file = $_FILES['foto_perfil'];
    
    // Validar tipo de arquivo
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
    $fileType = mime_content_type($file['tmp_name']);
    
    if (!in_array($fileType, $allowedTypes)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Apenas arquivos JPG, PNG ou GIF são permitidos'
        ]);
        exit;
    }
    
    // Validar tamanho (3MB)
    $maxSize = 3 * 1024 * 1024;
    if ($file['size'] > $maxSize) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'A imagem deve ter no máximo 3MB'
        ]);
        exit;
    }
    
    // Criar pasta uploads se não existir
    $uploadDir = __DIR__ . '/../uploads/';
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Gerar nome único para o arquivo
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $fileName = 'profile_' . $userId . '_' . time() . '.' . $extension;
    $uploadPath = $uploadDir . $fileName;
    
    // Mover arquivo
    if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
        $fotoPath = 'uploads/' . $fileName;
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Erro ao fazer upload da imagem'
        ]);
        exit;
    }
}

try {
    $conn = getConnection();
    
    // Buscar foto antiga antes de atualizar
    $stmt = $conn->prepare("SELECT foto_perfil FROM usuarios WHERE id = ?");
    $stmt->execute([$userId]);
    $oldUser = $stmt->fetch(PDO::FETCH_ASSOC);
    $oldFoto = $oldUser['foto_perfil'] ?? null;
    
    // Se tem nova foto, adicionar ao update
    if ($fotoPath) {
        $updateFields[] = 'foto_perfil = ?';
        $updateValues[] = $fotoPath;
        
        // Deletar foto antiga se existir
        if ($oldFoto && file_exists(__DIR__ . '/../' . $oldFoto)) {
            unlink(__DIR__ . '/../' . $oldFoto);
        }
    }
    
    // Se não há nada para atualizar
    if (empty($updateFields)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Nenhuma alteração para salvar'
        ]);
        exit;
    }
    
    // Marcar perfil como completo se tem idade e sexo
    if ($idade !== null && $sexo !== null) {
        $updateFields[] = 'perfil_completo = 1';
    }
    
    // Construir e executar query
    $updateValues[] = $userId; // ID para o WHERE
    $sql = "UPDATE usuarios SET " . implode(', ', $updateFields) . " WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->execute($updateValues);
    
    // Buscar dados atualizados do usuário
    $stmt = $conn->prepare("SELECT nome, foto_perfil, idade, sexo FROM usuarios WHERE id = ?");
    $stmt->execute([$userId]);
    $updatedUser = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Atualizar sessão
    if ($nome) $_SESSION['nome'] = $nome;
    if ($fotoPath) $_SESSION['foto_perfil'] = $fotoPath;
    if ($idade && $sexo) $_SESSION['perfil_completo'] = true;
    
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Perfil atualizado com sucesso!',
        'user' => $updatedUser
    ]);
    
} catch(PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao atualizar perfil: ' . $e->getMessage()
    ]);
}
?>
