<?php
/**
 * API para enviar mensagem
 */

session_start();

require_once 'config.php';

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Não autenticado']);
    exit;
}

$remetente_id = $_SESSION['user_id'];
$destinatario_id = isset($_POST['destinatario_id']) ? intval($_POST['destinatario_id']) : 0;
$mensagem = isset($_POST['mensagem']) ? trim($_POST['mensagem']) : '';
$tipo_midia = isset($_POST['tipo_midia']) ? $_POST['tipo_midia'] : null;
$caminho_midia = isset($_POST['caminho_midia']) ? $_POST['caminho_midia'] : null;
$reply_to_id = isset($_POST['reply_to_id']) ? intval($_POST['reply_to_id']) : null;

// Validar: precisa ter destinatário e (mensagem OU mídia)
if (!$destinatario_id) {
    echo json_encode(['success' => false, 'message' => 'Destinatário inválido']);
    exit;
}

if (!$mensagem && !$caminho_midia) {
    echo json_encode(['success' => false, 'message' => 'Mensagem ou mídia obrigatória']);
    exit;
}

// Se tiver mídia mas não tiver mensagem, deixar mensagem vazia
if ($caminho_midia && !$mensagem) {
    $mensagem = '';
}

try {
    $conn = getConnection();
    
    // Verificar palavras proibidas apenas em mensagens de texto
    if (!empty($mensagem) && !$caminho_midia) {
        require_once 'verificar_palavras.php';
        $resultado = verificarPalavrasProibidas($mensagem, $remetente_id, $destinatario_id);
        
        if (!$resultado['permitido']) {
            http_response_code(403);
            echo json_encode([
                'success' => false,
                'message' => $resultado['mensagem'],
                'banido' => $resultado['banido'] ?? false,
                'avisos' => $resultado['avisos'] ?? 0,
                'palavra_detectada' => $resultado['palavra'] ?? ''
            ]);
            exit;
        }
    }
    
    // Verificar saldo e membership do usuário
    $stmt = $conn->prepare("SELECT saldo, is_member FROM usuarios WHERE id = ?");
    $stmt->execute([$remetente_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Se não é membro e tem R$50 ou mais, bloquear
    if ($user && $user['saldo'] >= 50.00 && $user['is_member'] == 0) {
        echo json_encode([
            'success' => false, 
            'message' => 'Limite atingido. Torne-se membro para continuar.',
            'requires_membership' => true,
            'saldo' => number_format($user['saldo'], 2, ',', '.')
        ]);
        exit;
    }
    
    // Verificar se está bloqueado (se tabela existir)
    try {
        $stmt = $conn->prepare("
            SELECT id FROM bloqueios 
            WHERE blocked_user_id = ? AND user_id = ?
        ");
        $stmt->execute([$remetente_id, $destinatario_id]);
        
        if ($stmt->fetch()) {
            echo json_encode(['success' => false, 'message' => 'Este usuário bloqueou você']);
            exit;
        }
    } catch (PDOException $e) {
        // Tabela bloqueios ainda não existe, continuar normalmente
    }
    
    // Buscar ou criar conversa
    $user1 = min($remetente_id, $destinatario_id);
    $user2 = max($remetente_id, $destinatario_id);
    
    $stmt = $conn->prepare("
        SELECT id FROM conversas 
        WHERE user1_id = ? AND user2_id = ?
    ");
    $stmt->execute([$user1, $user2]);
    $conversa = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$conversa) {
        $stmt = $conn->prepare("
            INSERT INTO conversas (user1_id, user2_id) 
            VALUES (?, ?)
        ");
        $stmt->execute([$user1, $user2]);
        $conversa_id = $conn->lastInsertId();
    } else {
        $conversa_id = $conversa['id'];
    }
    
    // Inserir mensagem
    $stmt = $conn->prepare("
        INSERT INTO mensagens (conversa_id, remetente_id, destinatario_id, mensagem, tipo_midia, caminho_midia, reply_to_id) 
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $conversa_id,
        $remetente_id,
        $destinatario_id,
        $mensagem,
        $tipo_midia,
        $caminho_midia,
        $reply_to_id
    ]);
    
    $mensagem_id = $conn->lastInsertId();
    
    // Verificar cooldown de ganho (10 segundos desde última recompensa)
    $session_key = 'ultimo_ganho_' . $conversa_id;
    $pode_ganhar = true;
    
    if (isset($_SESSION[$session_key])) {
        $tempo_decorrido = time() - $_SESSION[$session_key];
        if ($tempo_decorrido < 10) {
            $pode_ganhar = false;
        }
    }
    
    // Buscar status de membro
    $stmt = $conn->prepare("SELECT is_member FROM usuarios WHERE id = ?");
    $stmt->execute([$remetente_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $is_member = $user_data['is_member'] ?? 0;
    
    // Definir valor do ganho: R$5 para membros, R$2 para não-membros
    $valor_ganho = $is_member ? 5.00 : 2.00;
    
    // Adicionar saldo apenas se pode ganhar
    if ($pode_ganhar) {
        $stmt = $conn->prepare("UPDATE usuarios SET saldo = saldo + ? WHERE id = ?");
        $stmt->execute([$valor_ganho, $remetente_id]);
        
        // Atualizar timestamp do último ganho
        $_SESSION[$session_key] = time();
    }
    
    // Buscar saldo atualizado
    $stmt = $conn->prepare("SELECT saldo FROM usuarios WHERE id = ?");
    $stmt->execute([$remetente_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    $novo_saldo = $user['saldo'];
    
    echo json_encode([
        'success' => true,
        'mensagem_id' => $mensagem_id,
        'created_at' => date('Y-m-d H:i:s'),
        'saldo' => number_format($novo_saldo, 2, ',', '.'),
        'ganhou' => $pode_ganhar,
        'ganho' => $pode_ganhar ? number_format($valor_ganho, 2, ',', '.') : '0,00'
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro: ' . $e->getMessage()
    ]);
}
?>
