<?php
/**
 * API para enviar presente (transferir saldo entre usuários)
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once 'config_global.php';

header('Content-Type: application/json; charset=utf-8');

initSession();

// Verificar autenticação
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Usuário não autenticado'
    ]);
    exit;
}

// Verificar método POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
    exit;
}

$remetente_id = $_SESSION['user_id'];
$data = json_decode(file_get_contents('php://input'), true);

// Validar dados
if (!isset($data['destinatario_id']) || !isset($data['valor'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Dados incompletos'
    ]);
    exit;
}

$destinatario_id = intval($data['destinatario_id']);
$valor = floatval($data['valor']);
$mensagem = isset($data['mensagem']) ? trim($data['mensagem']) : '';

// Validar valor
if ($valor <= 0) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Valor inválido'
    ]);
    exit;
}

// Não pode enviar para si mesmo
if ($remetente_id === $destinatario_id) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Não é possível enviar presente para você mesmo'
    ]);
    exit;
}

try {
    $conn = getConnection();
    $conn->beginTransaction();
    
    // Verificar saldo do remetente
    $stmt = $conn->prepare("SELECT saldo FROM usuarios WHERE id = ? FOR UPDATE");
    $stmt->execute([$remetente_id]);
    $saldo_remetente = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$saldo_remetente) {
        $conn->rollBack();
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Usuário não encontrado'
        ]);
        exit;
    }
    
    $saldo_atual = floatval($saldo_remetente['saldo']);
    
    // Verificar se tem saldo suficiente
    if ($saldo_atual < $valor) {
        $conn->rollBack();
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Saldo insuficiente'
        ]);
        exit;
    }
    
    // Deduzir saldo do remetente
    $novo_saldo_remetente = $saldo_atual - $valor;
    $stmt = $conn->prepare("UPDATE usuarios SET saldo = ? WHERE id = ?");
    $stmt->execute([$novo_saldo_remetente, $remetente_id]);
    
    // Adicionar saldo ao destinatário
    $stmt = $conn->prepare("UPDATE usuarios SET saldo = saldo + ? WHERE id = ?");
    $stmt->execute([$valor, $destinatario_id]);
    
    // Criar ou obter conversa
    $stmt = $conn->prepare("
        SELECT id FROM conversas 
        WHERE (user1_id = ? AND user2_id = ?) 
           OR (user1_id = ? AND user2_id = ?)
    ");
    $stmt->execute([$remetente_id, $destinatario_id, $destinatario_id, $remetente_id]);
    $conversa = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$conversa) {
        // Criar nova conversa
        $stmt = $conn->prepare("
            INSERT INTO conversas (user1_id, user2_id) 
            VALUES (?, ?)
        ");
        $stmt->execute([$remetente_id, $destinatario_id]);
        $conversa_id = $conn->lastInsertId();
    } else {
        $conversa_id = $conversa['id'];
    }
    
    // Salvar mensagem de presente
    $stmt = $conn->prepare("
        INSERT INTO mensagens (conversa_id, remetente_id, destinatario_id, mensagem, tipo_midia, gift_amount, created_at)
        VALUES (?, ?, ?, ?, 'gift', ?, NOW())
    ");
    $stmt->execute([$conversa_id, $remetente_id, $destinatario_id, $mensagem, $valor]);
    $mensagem_id = $conn->lastInsertId();
    
    // Registrar transação no histórico
    $stmt = $conn->prepare("
        INSERT INTO saldo_historico (user_id, tipo, valor, descricao, created_at)
        VALUES (?, 'gift_sent', ?, ?, NOW())
    ");
    $descricao_remetente = "Presente enviado para usuário #$destinatario_id" . ($mensagem ? " - $mensagem" : "");
    $stmt->execute([$remetente_id, -$valor, $descricao_remetente]);
    
    $stmt = $conn->prepare("
        INSERT INTO saldo_historico (user_id, tipo, valor, descricao, created_at)
        VALUES (?, 'gift_received', ?, ?, NOW())
    ");
    $descricao_destinatario = "Presente recebido de usuário #$remetente_id" . ($mensagem ? " - $mensagem" : "");
    $stmt->execute([$destinatario_id, $valor, $descricao_destinatario]);
    
    $conn->commit();
    
    echo json_encode([
        'success' => true,
        'message' => 'Presente enviado com sucesso!',
        'mensagem_id' => $mensagem_id,
        'novo_saldo' => number_format($novo_saldo_remetente, 2, '.', '')
    ]);
    
} catch (Exception $e) {
    if (isset($conn) && $conn->inTransaction()) {
        $conn->rollBack();
    }
    
    error_log("Erro ao enviar presente: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao processar presente: ' . $e->getMessage()
    ]);
}
