<?php
/**
 * API para cadastrar/atualizar chave PIX do usuário
 */

require_once 'config_global.php';
initSession();

// Verificar se está logado
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Usuário não autenticado'
    ]);
    exit;
}

// Verificar método
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
    exit;
}

// Receber dados
$tipo_chave = $_POST['tipo_chave'] ?? '';
$chave_pix = $_POST['chave_pix'] ?? '';
$user_id = $_SESSION['user_id'];

// Validar campos
if (empty($tipo_chave) || empty($chave_pix)) {
    echo json_encode([
        'success' => false,
        'message' => 'Preencha todos os campos'
    ]);
    exit;
}

// Validar tipo de chave
$tipos_validos = ['cpf', 'cnpj', 'email', 'telefone', 'aleatorio'];
if (!in_array($tipo_chave, $tipos_validos)) {
    echo json_encode([
        'success' => false,
        'message' => 'Tipo de chave inválido'
    ]);
    exit;
}

// Limpar chave PIX
$chave_pix = trim($chave_pix);

// Validações específicas por tipo de chave
switch ($tipo_chave) {
    case 'cpf':
        $chave_pix = preg_replace('/[^0-9]/', '', $chave_pix);
        if (strlen($chave_pix) !== 11) {
            echo json_encode([
                'success' => false,
                'message' => 'CPF deve ter 11 dígitos'
            ]);
            exit;
        }
        break;
    
    case 'cnpj':
        $chave_pix = preg_replace('/[^0-9]/', '', $chave_pix);
        if (strlen($chave_pix) !== 14) {
            echo json_encode([
                'success' => false,
                'message' => 'CNPJ deve ter 14 dígitos'
            ]);
            exit;
        }
        break;
    
    case 'email':
        if (!filter_var($chave_pix, FILTER_VALIDATE_EMAIL)) {
            echo json_encode([
                'success' => false,
                'message' => 'E-mail inválido'
            ]);
            exit;
        }
        break;
    
    case 'telefone':
        $chave_pix = preg_replace('/[^0-9]/', '', $chave_pix);
        if (strlen($chave_pix) < 10 || strlen($chave_pix) > 11) {
            echo json_encode([
                'success' => false,
                'message' => 'Telefone inválido'
            ]);
            exit;
        }
        break;
    
    case 'aleatorio':
        if (strlen($chave_pix) < 30) {
            echo json_encode([
                'success' => false,
                'message' => 'Chave aleatória inválida'
            ]);
            exit;
        }
        break;
}

try {
    $conn = getConnection();
    
    // Atualizar chave PIX
    $stmt = $conn->prepare("
        UPDATE usuarios 
        SET tipo_chave_pix = :tipo_chave, 
            chave_pix = :chave_pix,
            updated_at = NOW()
        WHERE id = :user_id
    ");
    
    $stmt->execute([
        ':tipo_chave' => $tipo_chave,
        ':chave_pix' => $chave_pix,
        ':user_id' => $user_id
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Chave PIX cadastrada com sucesso',
        'data' => [
            'tipo_chave' => $tipo_chave,
            'chave_pix' => $chave_pix
        ]
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao cadastrar chave PIX: ' . $e->getMessage()
    ]);
}
?>
