<?php
/**
 * API para recuperação de senha
 */

require_once 'config.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
    exit;
}

$data = json_decode(file_get_contents('php://input'), true);

// Validação
if (empty($data['whatsapp']) || empty($data['nova_senha'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'WhatsApp e nova senha são obrigatórios'
    ]);
    exit;
}

$whatsapp = preg_replace('/\D/', '', $data['whatsapp']);
$novaSenha = $data['nova_senha'];

// Validação da senha
if (strlen($novaSenha) < 8) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'A nova senha deve ter no mínimo 8 caracteres'
    ]);
    exit;
}

// Validação do WhatsApp
if (strlen($whatsapp) !== 11) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'WhatsApp inválido. Deve conter 11 dígitos'
    ]);
    exit;
}

try {
    $conn = getConnection();
    
    // Verificar se o WhatsApp existe
    $stmt = $conn->prepare("SELECT id FROM usuarios WHERE whatsapp = ?");
    $stmt->execute([$whatsapp]);
    $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$usuario) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => 'WhatsApp não encontrado'
        ]);
        exit;
    }
    
    // Hash da nova senha
    $senhaHash = password_hash($novaSenha, PASSWORD_BCRYPT);
    
    // Atualizar senha
    $stmt = $conn->prepare("UPDATE usuarios SET senha = ? WHERE id = ?");
    $stmt->execute([$senhaHash, $usuario['id']]);
    
    // Invalidar todos os tokens de login deste usuário
    $stmt = $conn->prepare("DELETE FROM login_tokens WHERE user_id = ?");
    $stmt->execute([$usuario['id']]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Senha alterada com sucesso! Faça login com sua nova senha.'
    ]);
    
} catch(PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao alterar senha: ' . $e->getMessage()
    ]);
}
?>
