<?php
/**
 * API para solicitar saque
 */

require_once 'config_global.php';
initSession();

// Verificar se está logado
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Usuário não autenticado'
    ]);
    exit;
}

// Verificar método
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
    exit;
}

// Receber dados
$valor = $_POST['valor'] ?? 0;
$user_id = $_SESSION['user_id'];

// Validar valor
$valor = floatval($valor);

if ($valor < 100) {
    echo json_encode([
        'success' => false,
        'message' => 'Valor mínimo para saque é R$ 100,00'
    ]);
    exit;
}

try {
    $conn = getConnection();
    
    // Buscar dados do usuário
    $stmt = $conn->prepare("SELECT saldo, chave_pix FROM usuarios WHERE id = :user_id");
    $stmt->execute([':user_id' => $user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        echo json_encode([
            'success' => false,
            'message' => 'Usuário não encontrado'
        ]);
        exit;
    }
    
    // Verificar se tem chave PIX
    if (empty($user['chave_pix'])) {
        echo json_encode([
            'success' => false,
            'message' => 'Cadastre uma chave PIX antes de solicitar saque'
        ]);
        exit;
    }
    
    // Verificar saldo
    if ($valor > $user['saldo']) {
        echo json_encode([
            'success' => false,
            'message' => 'Saldo insuficiente'
        ]);
        exit;
    }
    
    // Criar tabela de saques se não existir
    $conn->exec("
        CREATE TABLE IF NOT EXISTS saques (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            valor DECIMAL(10,2) NOT NULL,
            status ENUM('pendente', 'processando', 'aprovado', 'rejeitado') DEFAULT 'pendente',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES usuarios(id)
        )
    ");
    
    // Iniciar transação
    $conn->beginTransaction();
    
    // Registrar saque
    $stmt = $conn->prepare("
        INSERT INTO saques (user_id, valor, status) 
        VALUES (:user_id, :valor, 'pendente')
    ");
    $stmt->execute([
        ':user_id' => $user_id,
        ':valor' => $valor
    ]);
    
    // Descontar saldo
    $stmt = $conn->prepare("
        UPDATE usuarios 
        SET saldo = saldo - :valor,
            updated_at = NOW()
        WHERE id = :user_id
    ");
    $stmt->execute([
        ':valor' => $valor,
        ':user_id' => $user_id
    ]);
    
    // Confirmar transação
    $conn->commit();
    
    // Calcular novo saldo
    $novo_saldo = $user['saldo'] - $valor;
    
    echo json_encode([
        'success' => true,
        'message' => 'Saque solicitado com sucesso! Processaremos em até 24 horas.',
        'data' => [
            'valor' => $valor,
            'novo_saldo' => $novo_saldo
        ]
    ]);
    
} catch (PDOException $e) {
    // Reverter transação em caso de erro
    if ($conn->inTransaction()) {
        $conn->rollBack();
    }
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao processar saque: ' . $e->getMessage()
    ]);
}
?>
