<?php
/**
 * API de Cadastro de Usuários
 */

// Habilitar logs de erro para debug
error_reporting(E_ALL);
ini_set('display_errors', 0); // Não mostrar na tela para não quebrar o JSON
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/php_errors.log');

require_once 'config_global.php';

// Iniciar sessão ANTES de qualquer output
initSession();

// Log de debug
$debugLog = [
    'timestamp' => date('Y-m-d H:i:s'),
    'method' => $_SERVER['REQUEST_METHOD'],
    'raw_input' => file_get_contents('php://input')
];
file_put_contents(__DIR__ . '/register_debug.log', json_encode($debugLog, JSON_PRETTY_PRINT) . "\n", FILE_APPEND);

// Verifica se é POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
    exit;
}

// Pega os dados do JSON
$data = json_decode(file_get_contents('php://input'), true);

// Validação dos dados
if (empty($data['nome']) || empty($data['whatsapp']) || empty($data['senha'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Todos os campos são obrigatórios'
    ]);
    exit;
}

$nome = trim($data['nome']);
$whatsapp = preg_replace('/\D/', '', $data['whatsapp']); // Remove caracteres não numéricos
$senha = $data['senha'];

// Validação do WhatsApp (deve ter 11 dígitos)
if (strlen($whatsapp) !== 11) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'WhatsApp inválido. Deve conter 11 dígitos'
    ]);
    exit;
}

// Validação da senha (mínimo 8 caracteres)
if (strlen($senha) < 8) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'A senha deve ter no mínimo 8 caracteres'
    ]);
    exit;
}

try {
    $conn = getConnection();
    
    // Verifica se o WhatsApp já existe
    $stmt = $conn->prepare("SELECT id FROM usuarios WHERE whatsapp = ?");
    $stmt->execute([$whatsapp]);
    
    if ($stmt->fetch()) {
        http_response_code(409);
        echo json_encode([
            'success' => false,
            'message' => 'Este WhatsApp já está cadastrado'
        ]);
        exit;
    }
    
    // Hash da senha usando bcrypt
    $senhaHash = password_hash($senha, PASSWORD_BCRYPT);
    
    // Insere o usuário
    $stmt = $conn->prepare("
        INSERT INTO usuarios (nome, whatsapp, senha) 
        VALUES (?, ?, ?)
    ");
    
    $stmt->execute([$nome, $whatsapp, $senhaHash]);
    
    $userId = $conn->lastInsertId();
    
    // Definir variáveis de sessão
    $_SESSION['user_id'] = $userId;
    $_SESSION['user_nome'] = $nome;
    $_SESSION['user_whatsapp'] = $whatsapp;
    
    http_response_code(201);
    echo json_encode([
        'success' => true,
        'message' => 'Cadastro realizado com sucesso!',
        'redirect' => getUrl('complete_profile/index.html'),
        'user' => [
            'id' => $userId,
            'nome' => $nome,
            'whatsapp' => $whatsapp
        ]
    ]);
    
} catch(PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao cadastrar usuário: ' . $e->getMessage()
    ]);
}
?>
