<?php
/**
 * API de Login de Usuários
 */

require_once 'config_global.php';

// Iniciar sessão ANTES de qualquer validação
initSession();

// Verifica se é POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
    exit;
}

// Pega os dados do JSON
$data = json_decode(file_get_contents('php://input'), true);

// Validação dos dados
if (empty($data['whatsapp']) || empty($data['senha'])) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'WhatsApp e senha são obrigatórios'
    ]);
    exit;
}

$whatsapp = preg_replace('/\D/', '', $data['whatsapp']); // Remove caracteres não numéricos
$senha = $data['senha'];
$remember = isset($data['remember']) ? $data['remember'] : false;

try {
    $conn = getConnection();
    
    // Busca o usuário pelo WhatsApp
    $stmt = $conn->prepare("
        SELECT id, nome, whatsapp, senha, perfil_completo, bloqueado 
        FROM usuarios 
        WHERE whatsapp = ?
    ");
    
    $stmt->execute([$whatsapp]);
    $usuario = $stmt->fetch();
    
    // Verifica se o usuário existe
    if (!$usuario) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'WhatsApp ou senha incorretos'
        ]);
        exit;
    }
    
    // Verifica se a conta está bloqueada
    if (isset($usuario['bloqueado']) && $usuario['bloqueado'] == 1) {
        http_response_code(403);
        echo json_encode([
            'success' => false,
            'message' => 'Sua conta foi bloqueada. Entre em contato com o suporte.',
            'blocked' => true
        ]);
        exit;
    }
    
    // Verifica a senha
    if (!password_verify($senha, $usuario['senha'])) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'WhatsApp ou senha incorretos'
        ]);
        exit;
    }
    
    // Definir variáveis de sessão
    $_SESSION['user_id'] = $usuario['id'];
    $_SESSION['user_nome'] = $usuario['nome'];
    $_SESSION['user_whatsapp'] = $usuario['whatsapp'];
    
    // Se "lembrar login" estiver marcado, criar cookie por 24 horas
    if ($remember) {
        // Gerar token único
        $token = bin2hex(random_bytes(32));
        $expires = time() + (24 * 60 * 60); // 24 horas
        
        // Salvar token no banco
        $stmt = $conn->prepare("
            INSERT INTO login_tokens (user_id, token, expires_at)
            VALUES (?, ?, FROM_UNIXTIME(?))
            ON DUPLICATE KEY UPDATE token = ?, expires_at = FROM_UNIXTIME(?)
        ");
        $stmt->execute([$usuario['id'], $token, $expires, $token, $expires]);
        
        // Criar cookie
        setcookie('remember_token', $token, $expires, '/', '', false, true); // httpOnly = true
    }
    
    // Determinar para onde redirecionar
    $redirect = $usuario['perfil_completo'] 
        ? getUrl('principal/index.html')
        : getUrl('complete_profile/index.html');
    
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Login realizado com sucesso!',
        'perfil_completo' => (bool)$usuario['perfil_completo'],
        'redirect' => $redirect,
        'user' => [
            'id' => $usuario['id'],
            'nome' => $usuario['nome'],
            'whatsapp' => $usuario['whatsapp']
        ]
    ]);
    
} catch(PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao fazer login: ' . $e->getMessage()
    ]);
}
?>
