<?php
/**
 * Classe para gerenciar o tracking de eventos para Kwai Ads
 */

require_once 'kwai_config.php';
require_once 'config.php';

class KwaiTracking {
    private $conn;
    private $pixelId;
    private $accessToken;
    
    public function __construct($conn) {
        $this->conn = $conn;
        $this->pixelId = KWAI_PIXEL_ID;
        $this->accessToken = KWAI_WEB_EVENT_TOKEN;
    }
    
    /**
     * Salva o click_id associado a um usuário
     */
    public function saveClickId($userId, $clickId) {
        try {
            // Atualiza ou insere o click_id para o usuário
            $stmt = $this->conn->prepare("
                INSERT INTO kwai_tracking (user_id, click_id, created_at, updated_at) 
                VALUES (?, ?, NOW(), NOW())
                ON DUPLICATE KEY UPDATE 
                    click_id = VALUES(click_id),
                    updated_at = NOW()
            ");
            $stmt->bind_param("is", $userId, $clickId);
            $stmt->execute();
            return true;
        } catch (Exception $e) {
            error_log("Erro ao salvar click_id: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Recupera o click_id de um usuário
     */
    public function getClickId($userId) {
        try {
            $stmt = $this->conn->prepare("SELECT click_id FROM kwai_tracking WHERE user_id = ?");
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($row = $result->fetch_assoc()) {
                return $row['click_id'];
            }
            return null;
        } catch (Exception $e) {
            error_log("Erro ao recuperar click_id: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Envia um evento para a API do Kwai
     */
    public function sendEvent($eventName, $clickId, $properties = [], $value = null, $currency = null) {
        // Valida se o evento é suportado
        if (!in_array($eventName, KWAI_EVENTS)) {
            error_log("Evento não suportado: $eventName");
            return false;
        }
        
        // Prepara os dados do evento
        $eventData = [
            'access_token' => $this->accessToken,
            'clickid' => $clickId,
            'event_name' => $eventName,
            'is_attributed' => KWAI_IS_ATTRIBUTED,
            'mmpcode' => KWAI_MMP_CODE,
            'pixelId' => $this->pixelId,
            'pixelSdkVersion' => KWAI_SDK_VERSION,
            'testFlag' => KWAI_TEST_MODE,
            'trackFlag' => KWAI_TRACK_MODE
        ];
        
        // Adiciona valor e moeda se fornecidos
        if ($value !== null) {
            $eventData['value'] = strval($value);
        }
        if ($currency !== null) {
            $eventData['currency'] = $currency;
        } else if ($value !== null) {
            $eventData['currency'] = KWAI_DEFAULT_CURRENCY;
        }
        
        // Adiciona properties se fornecidas
        if (!empty($properties)) {
            // Adiciona timestamp se não fornecido
            if (!isset($properties['event_timestamp'])) {
                $properties['event_timestamp'] = time();
            }
            $eventData['properties'] = json_encode($properties);
        }
        
        // Envia a requisição
        return $this->sendPostRequest(KWAI_API_POST_URL, $eventData);
    }
    
    /**
     * Envia requisição POST para a API do Kwai
     */
    private function sendPostRequest($url, $data) {
        $ch = curl_init($url);
        
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Accept: application/json;charset=utf-8'
            ],
            CURLOPT_TIMEOUT => 10
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        // Log da requisição
        error_log("Kwai API Request: " . json_encode($data));
        error_log("Kwai API Response ($httpCode): $response");
        
        if ($error) {
            error_log("Erro cURL ao enviar evento Kwai: $error");
            return false;
        }
        
        // Processa resposta
        $responseData = json_decode($response, true);
        
        if ($responseData && isset($responseData['result'])) {
            if ($responseData['result'] === 1) {
                return true;
            } else {
                error_log("Erro na API Kwai: " . $responseData['error_msg']);
                return false;
            }
        }
        
        return $httpCode >= 200 && $httpCode < 300;
    }
    
    /**
     * Métodos auxiliares para eventos específicos
     */
    
    public function trackRegistration($userId, $clickId) {
        return $this->sendEvent(
            KWAI_EVENTS['COMPLETE_REGISTRATION'],
            $clickId,
            ['content_name' => 'User Registration']
        );
    }
    
    public function trackPurchase($userId, $clickId, $value, $orderId = null, $products = []) {
        $properties = [
            'content_type' => 'product'
        ];
        
        if ($orderId) {
            $properties['content_id'] = $orderId;
        }
        
        if (!empty($products)) {
            $properties['contents'] = $products;
        }
        
        return $this->sendEvent(
            KWAI_EVENTS['PURCHASE'],
            $clickId,
            $properties,
            $value,
            KWAI_DEFAULT_CURRENCY
        );
    }
    
    public function trackInitiatedCheckout($userId, $clickId, $value = null) {
        return $this->sendEvent(
            KWAI_EVENTS['INITIATED_CHECKOUT'],
            $clickId,
            ['content_name' => 'Checkout Started'],
            $value,
            KWAI_DEFAULT_CURRENCY
        );
    }
    
    public function trackAddPaymentInfo($userId, $clickId) {
        return $this->sendEvent(
            KWAI_EVENTS['ADD_PAYMENT_INFO'],
            $clickId,
            ['content_name' => 'Payment Info Added']
        );
    }
    
    public function trackPageView($userId, $clickId, $pageName) {
        return $this->sendEvent(
            KWAI_EVENTS['CONTENT_VIEW'],
            $clickId,
            ['content_name' => $pageName]
        );
    }
    
    public function trackFirstDeposit($userId, $clickId, $value) {
        return $this->sendEvent(
            KWAI_EVENTS['FIRST_DEPOSIT'],
            $clickId,
            ['content_name' => 'First Deposit'],
            $value,
            KWAI_DEFAULT_CURRENCY
        );
    }
    
    /**
     * Registra evento no banco de dados local para histórico
     */
    public function logEventToDatabase($userId, $eventName, $clickId, $properties = [], $success = false) {
        // Se não tem userId, usa 0 como placeholder
        if (!$userId) {
            $userId = 0;
        }
        
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO kwai_events_log 
                (user_id, event_name, click_id, properties, success, created_at) 
                VALUES (?, ?, ?, ?, ?, NOW())
            ");
            
            $propertiesJson = json_encode($properties);
            $successInt = $success ? 1 : 0;
            
            $stmt->bind_param("isssi", $userId, $eventName, $clickId, $propertiesJson, $successInt);
            $stmt->execute();
            return true;
        } catch (Exception $e) {
            error_log("Erro ao registrar evento no banco: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Envia evento e registra no banco
     */
    public function trackAndLog($userId, $eventName, $properties = [], $value = null, $currency = null) {
        $clickId = $this->getClickId($userId);
        
        if (!$clickId) {
            error_log("Click ID não encontrado para usuário $userId");
            return false;
        }
        
        $success = $this->sendEvent($eventName, $clickId, $properties, $value, $currency);
        $this->logEventToDatabase($userId, $eventName, $clickId, $properties, $success);
        
        return $success;
    }
}
