<?php
/**
 * Endpoint para enviar eventos específicos para a API do Kwai
 */

require_once 'config_global.php';
require_once 'kwai_tracking.php';

header('Content-Type: application/json');

// Iniciar sessão com configurações corretas
initSession();

try {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        throw new Exception('Dados inválidos');
    }
    
    $eventType = $data['event_type'] ?? null;
    
    // Para eventos de visualização, permite sem login (apenas com click_id na sessão)
    $allowWithoutLogin = ['page_view', 'content_view'];
    
    if (!in_array($eventType, $allowWithoutLogin) && !isset($_SESSION['user_id'])) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Usuário não autenticado'
        ]);
        exit;
    }
    
    $tracking = new KwaiTracking($conn);
    $clickId = null;
    $userId = null;
    
    // Tenta obter click_id
    if (isset($_SESSION['user_id'])) {
        $userId = $_SESSION['user_id'];
        $clickId = $tracking->getClickId($userId);
    }
    
    // Se não tem no banco, tenta da sessão
    if (!$clickId && isset($_SESSION['kwai_click_id'])) {
        $clickId = $_SESSION['kwai_click_id'];
        
        // Se usuário está logado, salva no banco
        if ($userId) {
            $tracking->saveClickId($userId, $clickId);
        }
    }
    
    // Se não tem click_id, não pode enviar evento
    if (!$clickId) {
        echo json_encode([
            'success' => false,
            'message' => 'Click ID não encontrado. Usuário não veio de anúncio Kwai.',
            'info' => 'Este evento só é enviado para usuários que chegaram via anúncio Kwai'
        ]);
        exit;
    }
    
    $success = false;
    $message = '';
    
    switch ($eventType) {
        case 'registration':
            // Evento de registro completo
            if (!$userId) {
                throw new Exception('User ID necessário para evento de registro');
            }
            $success = $tracking->trackRegistration($userId, $clickId);
            $message = 'Evento de registro enviado';
            break;
            
        case 'purchase':
            // Evento de compra
            if (!$userId) {
                throw new Exception('User ID necessário para evento de compra');
            }
            $value = $data['value'] ?? 0;
            $orderId = $data['order_id'] ?? null;
            $products = $data['products'] ?? [];
            
            $success = $tracking->trackPurchase($userId, $clickId, $value, $orderId, $products);
            $message = 'Evento de compra enviado';
            break;
            
        case 'initiated_checkout':
            // Evento de início de checkout
            if (!$userId) {
                throw new Exception('User ID necessário para evento de checkout');
            }
            $value = $data['value'] ?? null;
            $success = $tracking->trackInitiatedCheckout($userId, $clickId, $value);
            $message = 'Evento de início de checkout enviado';
            break;
            
        case 'add_payment_info':
            // Evento de adição de informação de pagamento
            if (!$userId) {
                throw new Exception('User ID necessário para evento de pagamento');
            }
            $success = $tracking->trackAddPaymentInfo($userId, $clickId);
            $message = 'Evento de adição de pagamento enviado';
            break;
            
        case 'page_view':
        case 'content_view':
            // Evento de visualização de página (não precisa de userId)
            $pageName = $data['page_name'] ?? 'Unknown Page';
            $success = $tracking->sendEvent(
                KWAI_EVENTS['CONTENT_VIEW'],
                $clickId,
                ['content_name' => $pageName]
            );
            // Registra no log mesmo sem userId
            $tracking->logEventToDatabase($userId ?? 0, KWAI_EVENTS['CONTENT_VIEW'], $clickId, ['content_name' => $pageName], $success);
            $message = 'Evento de visualização de página enviado';
            break;
            
        case 'first_deposit':
            // Evento de primeiro depósito
            if (!$userId) {
                throw new Exception('User ID necessário para evento de depósito');
            }
            $value = $data['value'] ?? 0;
            $success = $tracking->trackFirstDeposit($userId, $clickId, $value);
            $message = 'Evento de primeiro depósito enviado';
            break;
            
        case 'custom':
            // Evento customizado
            $eventName = $data['event_name'] ?? null;
            $properties = $data['properties'] ?? [];
            $value = $data['value'] ?? null;
            $currency = $data['currency'] ?? null;
            
            if (!$eventName) {
                throw new Exception('Nome do evento é obrigatório para eventos customizados');
            }
            
            $success = $tracking->sendEvent($eventName, $clickId, $properties, $value, $currency);
            $tracking->logEventToDatabase($userId ?? 0, $eventName, $clickId, $properties, $success);
            $message = 'Evento customizado enviado';
            break;
            
        default:
            throw new Exception('Tipo de evento não suportado: ' . $eventType);
    }
    
    if ($success) {
        echo json_encode([
            'success' => true,
            'message' => $message
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Falha ao enviar evento para Kwai API'
        ]);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro: ' . $e->getMessage()
    ]);
}

$conn->close();
