<?php
/**
 * API para iniciar ou obter uma conversa entre dois usuários
 */

session_start();

require_once 'config.php';

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Não autenticado']);
    exit;
}

$current_user_id = $_SESSION['user_id'];
$other_user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;

if (!$other_user_id) {
    echo json_encode(['success' => false, 'message' => 'ID do usuário não fornecido']);
    exit;
}

try {
    $conn = getConnection();
    
    // Verificar se conversa já existe (em qualquer ordem)
    $stmt = $conn->prepare("
        SELECT id FROM conversas 
        WHERE (user1_id = ? AND user2_id = ?)
        OR (user1_id = ? AND user2_id = ?)
    ");
    $stmt->execute([
        $current_user_id, $other_user_id,
        $other_user_id, $current_user_id
    ]);
    
    $conversa = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($conversa) {
        // Conversa já existe
        $conversa_id = $conversa['id'];
    } else {
        // Criar nova conversa (sempre menor ID primeiro)
        $user1 = min($current_user_id, $other_user_id);
        $user2 = max($current_user_id, $other_user_id);
        
        $stmt = $conn->prepare("
            INSERT INTO conversas (user1_id, user2_id) 
            VALUES (?, ?)
        ");
        $stmt->execute([$user1, $user2]);
        
        $conversa_id = $conn->lastInsertId();
    }
    
    // Buscar dados do outro usuário
    $stmt = $conn->prepare("
        SELECT 
            id, 
            nome, 
            foto_perfil, 
            idade, 
            sexo,
            TIMESTAMPDIFF(SECOND, last_seen, NOW()) as segundos_offline
        FROM usuarios 
        WHERE id = ?
    ");
    $stmt->execute([$other_user_id]);
    $other_user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Adicionar status online (30 segundos de tolerância)
    $other_user['online'] = $other_user['segundos_offline'] <= 30;
    unset($other_user['segundos_offline']);
    
    echo json_encode([
        'success' => true,
        'conversa_id' => $conversa_id,
        'other_user' => $other_user
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro: ' . $e->getMessage()
    ]);
}
?>
