<?php
/**
 * API para buscar mensagens de uma conversa
 */

session_start();

require_once 'config.php';

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Não autenticado']);
    exit;
}

$current_user_id = $_SESSION['user_id'];
$conversa_id = isset($_GET['conversa_id']) ? intval($_GET['conversa_id']) : 0;
$last_id = isset($_GET['last_id']) ? intval($_GET['last_id']) : 0;

if (!$conversa_id) {
    echo json_encode(['success' => false, 'message' => 'ID da conversa não fornecido']);
    exit;
}

try {
    $conn = getConnection();
    
    // Buscar mensagens
    if ($last_id > 0) {
        // Buscar apenas mensagens novas
        $stmt = $conn->prepare("
            SELECT 
                m.id,
                m.remetente_id,
                m.destinatario_id,
                m.mensagem,
                m.tipo_midia,
                m.caminho_midia,
                m.gift_amount,
                m.lida,
                m.created_at,
                m.reply_to_id,
                u.nome as remetente_nome,
                u.foto_perfil as remetente_foto,
                rm.mensagem as replied_mensagem,
                rm.tipo_midia as replied_tipo_midia,
                rm.remetente_id as replied_remetente_id,
                ru.nome as replied_remetente_nome
            FROM mensagens m
            LEFT JOIN usuarios u ON m.remetente_id = u.id
            LEFT JOIN mensagens rm ON m.reply_to_id = rm.id
            LEFT JOIN usuarios ru ON rm.remetente_id = ru.id
            WHERE m.conversa_id = ?
            AND m.id > ?
            ORDER BY m.created_at ASC
        ");
        $stmt->execute([$conversa_id, $last_id]);
    } else {
        // Buscar todas as mensagens
        $stmt = $conn->prepare("
            SELECT 
                m.id,
                m.remetente_id,
                m.destinatario_id,
                m.mensagem,
                m.tipo_midia,
                m.caminho_midia,
                m.gift_amount,
                m.lida,
                m.created_at,
                m.reply_to_id,
                u.nome as remetente_nome,
                u.foto_perfil as remetente_foto,
                rm.mensagem as replied_mensagem,
                rm.tipo_midia as replied_tipo_midia,
                rm.remetente_id as replied_remetente_id,
                ru.nome as replied_remetente_nome
            FROM mensagens m
            LEFT JOIN usuarios u ON m.remetente_id = u.id
            LEFT JOIN mensagens rm ON m.reply_to_id = rm.id
            LEFT JOIN usuarios ru ON rm.remetente_id = ru.id
            WHERE m.conversa_id = ?
            ORDER BY m.created_at ASC
        ");
        $stmt->execute([$conversa_id]);
    }
    
    $mensagens = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Marcar como lidas as mensagens recebidas
    $stmt = $conn->prepare("
        UPDATE mensagens 
        SET lida = 1 
        WHERE conversa_id = ? 
        AND destinatario_id = ? 
        AND lida = 0
    ");
    $stmt->execute([$conversa_id, $current_user_id]);
    
    // Formatar mensagens
    foreach ($mensagens as &$msg) {
        $msg['is_mine'] = ($msg['remetente_id'] == $current_user_id);
        $msg['time'] = date('H:i', strtotime($msg['created_at']));
        
        // Adicionar tipo se for presente
        if ($msg['tipo_midia'] === 'gift') {
            $msg['tipo'] = 'gift';
            $msg['valor'] = floatval($msg['gift_amount']);
        }
        
        // Se há uma mensagem respondida, formatar os dados
        if ($msg['reply_to_id'] && $msg['replied_mensagem']) {
            $msg['replied_message'] = [
                'id' => $msg['reply_to_id'],
                'mensagem' => $msg['replied_mensagem'],
                'tipo_midia' => $msg['replied_tipo_midia'],
                'remetente_id' => $msg['replied_remetente_id'],
                'remetente_nome' => $msg['replied_remetente_nome']
            ];
        }
        
        // Remover colunas temporárias do resultado
        unset($msg['replied_mensagem']);
        unset($msg['replied_tipo_midia']);
        unset($msg['replied_remetente_id']);
        unset($msg['replied_remetente_nome']);
    }
    
    // Se é polling (last_id > 0), buscar também status de leitura das minhas mensagens
    $read_status = [];
    if ($last_id > 0) {
        $stmt = $conn->prepare("
            SELECT id
            FROM mensagens
            WHERE conversa_id = ?
            AND remetente_id = ?
            AND lida = 1
            ORDER BY id DESC
            LIMIT 50
        ");
        $stmt->execute([$conversa_id, $current_user_id]);
        $read_status = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);
    }
    
    echo json_encode([
        'success' => true,
        'mensagens' => $mensagens,
        'read_messages' => $read_status
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro: ' . $e->getMessage()
    ]);
}
?>
