<?php
/**
 * API para buscar lista de conversas do usuário
 */

session_start();

require_once 'config.php';

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Não autenticado']);
    exit;
}

$current_user_id = $_SESSION['user_id'];

try {
    $conn = getConnection();
    
    // Buscar conversas com última mensagem e contador de não lidas
    $stmt = $conn->prepare("
        SELECT 
            c.id as conversa_id,
            CASE 
                WHEN c.user1_id = ? THEN c.user2_id 
                ELSE c.user1_id 
            END as other_user_id,
            u.nome as other_user_nome,
            u.foto_perfil as other_user_foto,
            (SELECT mensagem FROM mensagens WHERE conversa_id = c.id ORDER BY created_at DESC LIMIT 1) as ultima_mensagem,
            (SELECT tipo_midia FROM mensagens WHERE conversa_id = c.id ORDER BY created_at DESC LIMIT 1) as ultima_tipo_midia,
            (SELECT remetente_id FROM mensagens WHERE conversa_id = c.id ORDER BY created_at DESC LIMIT 1) as ultima_mensagem_remetente,
            (SELECT created_at FROM mensagens WHERE conversa_id = c.id ORDER BY created_at DESC LIMIT 1) as ultima_mensagem_data,
            (SELECT COUNT(*) FROM mensagens WHERE conversa_id = c.id AND destinatario_id = ? AND lida = 0) as nao_lidas
        FROM conversas c
        LEFT JOIN usuarios u ON (
            CASE 
                WHEN c.user1_id = ? THEN c.user2_id 
                ELSE c.user1_id 
            END = u.id
        )
        WHERE c.user1_id = ? OR c.user2_id = ?
        ORDER BY ultima_mensagem_data DESC
    ");
    
    $stmt->execute([
        $current_user_id,
        $current_user_id,
        $current_user_id,
        $current_user_id,
        $current_user_id
    ]);
    $conversas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Formatar dados
    foreach ($conversas as &$conv) {
        $conv['nao_lidas'] = (int)$conv['nao_lidas'];
        
        // Determinar a mensagem a exibir
        $mensagemExibir = '';
        
        if ($conv['ultima_tipo_midia']) {
            // Se for mídia, mostrar ícone ao invés da mensagem
            if ($conv['ultima_tipo_midia'] === 'image') {
                $mensagemExibir = '📷 Imagem';
            } elseif ($conv['ultima_tipo_midia'] === 'video') {
                $mensagemExibir = '🎥 Vídeo';
            } elseif ($conv['ultima_tipo_midia'] === 'audio') {
                $mensagemExibir = '🎤 Áudio';
            }
        } else {
            // Se for texto normal
            $mensagemExibir = $conv['ultima_mensagem'] ?: 'Sem mensagens ainda';
        }
        
        // Adicionar prefixo "Você:" ou "Nome:"
        if ($conv['ultima_mensagem'] || $conv['ultima_tipo_midia']) {
            if ($conv['ultima_mensagem_remetente'] == $current_user_id) {
                $conv['ultima_mensagem_formatada'] = 'Você: ' . $mensagemExibir;
            } else {
                $conv['ultima_mensagem_formatada'] = $mensagemExibir;
            }
        } else {
            $conv['ultima_mensagem_formatada'] = 'Sem mensagens ainda';
        }
        
        if ($conv['ultima_mensagem_data']) {
            $timestamp = strtotime($conv['ultima_mensagem_data']);
            $hoje = strtotime('today');
            
            if ($timestamp >= $hoje) {
                $conv['ultima_mensagem_tempo'] = date('H:i', $timestamp);
            } else {
                $conv['ultima_mensagem_tempo'] = date('d/m/Y', $timestamp);
            }
        } else {
            $conv['ultima_mensagem_tempo'] = '';
        }
    }
    
    // Contar total de mensagens não lidas
    $stmt = $conn->prepare("
        SELECT COUNT(*) as total
        FROM mensagens 
        WHERE destinatario_id = ? 
        AND lida = 0
    ");
    $stmt->execute([$current_user_id]);
    $total_nao_lidas = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    echo json_encode([
        'success' => true,
        'conversas' => $conversas,
        'total_nao_lidas' => (int)$total_nao_lidas
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro: ' . $e->getMessage()
    ]);
}
?>
