<?php
/**
 * ============================================================================
 * CONFIGURAÇÃO GLOBAL DO SISTEMA
 * ============================================================================
 * 
 * Este é o ÚNICO arquivo que você precisa editar ao transferir para outro domínio!
 * 
 * INSTRUÇÕES DE TRANSFERÊNCIA:
 * 1. Altere as credenciais do banco de dados (DB_HOST, DB_NAME, DB_USER, DB_PASS)
 * 2. Altere o BASE_PATH se o sistema não estiver na raiz do domínio
 * 3. Execute http://seu-dominio.com/install.php para criar as tabelas
 * 4. Pronto! O sistema estará 100% funcional
 * 
 * ============================================================================
 */

// ============================================================================
// CONFIGURAÇÕES DO BANCO DE DADOS
// ============================================================================

define('DB_HOST', 'localhost');
define('DB_NAME', 'gabr8802_mensagem');
define('DB_USER', 'gabr8802_mensagem');
define('DB_PASS', 'Kakekikoku18@');

// ============================================================================
// CONFIGURAÇÕES DE PATHS E URLs
// ============================================================================

/**
 * BASE_PATH: Caminho base da aplicação (relativo à raiz do domínio)
 * 
 * Exemplos:
 * - Se o sistema está em: http://seudominio.com/
 *   Defina: define('BASE_PATH', '/');
 * 
 * - Se o sistema está em: http://seudominio.com/mensagem/
 *   Defina: define('BASE_PATH', '/mensagem/');
 * 
 * - Se o sistema está em: http://seudominio.com/app/chat/
 *   Defina: define('BASE_PATH', '/app/chat/');
 * 
 * IMPORTANTE: Sempre termine com /
 */
define('BASE_PATH', '/mensagem/');

/**
 * URL_BASE: URL completa do domínio (detectada automaticamente)
 */
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'] ?? 'localhost';
define('URL_BASE', $protocol . '://' . $host);

/**
 * URL_COMPLETA: URL completa da aplicação
 */
define('URL_COMPLETA', URL_BASE . rtrim(BASE_PATH, '/'));

/**
 * DIR_ROOT: Caminho absoluto do servidor para a raiz do projeto
 */
define('DIR_ROOT', dirname(__DIR__));

/**
 * DIR_BACKEND: Caminho absoluto para a pasta backend
 */
define('DIR_BACKEND', __DIR__);

/**
 * DIR_UPLOADS: Caminho absoluto para a pasta de uploads
 */
define('DIR_UPLOADS', DIR_ROOT . '/uploads');

/**
 * URL_UPLOADS: URL para acessar uploads
 */
define('URL_UPLOADS', BASE_PATH . 'uploads/');

/**
 * URL_ASSETS: URL para acessar assets
 */
define('URL_ASSETS', BASE_PATH . 'assets/');

/**
 * URL_BACKEND: URL para acessar backend
 */
define('URL_BACKEND', BASE_PATH . 'backend/');

// ============================================================================
// CONFIGURAÇÕES GERAIS
// ============================================================================

/**
 * Timezone padrão
 */
date_default_timezone_set('America/Sao_Paulo');

/**
 * Charset padrão
 */
define('CHARSET', 'utf8mb4');

/**
 * Environment (development ou production)
 */
define('ENVIRONMENT', 'production'); // Mude para 'development' para debug

/**
 * Configurar exibição de erros baseado no ambiente
 */
if (ENVIRONMENT === 'development') {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    ini_set('log_errors', 1);
    ini_set('error_log', DIR_BACKEND . '/error_log');
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    ini_set('error_log', DIR_BACKEND . '/error_log');
}

// ============================================================================
// CONFIGURAÇÕES DE SESSÃO
// ============================================================================

/**
 * Nome da sessão (único para esta aplicação)
 */
define('SESSION_NAME', 'mensagem_app_session');

/**
 * Tempo de vida da sessão em segundos (24 horas)
 */
define('SESSION_LIFETIME', 86400);

/**
 * Cookie settings para sessão
 */
define('SESSION_COOKIE_PATH', rtrim(BASE_PATH, '/') . '/');
define('SESSION_COOKIE_DOMAIN', ''); // Deixe vazio para auto-detect
define('SESSION_COOKIE_HTTPONLY', true);

// ============================================================================
// FUNÇÃO DE CONEXÃO COM BANCO DE DADOS (PDO)
// ============================================================================

/**
 * Retorna uma conexão PDO com o banco de dados
 * 
 * @return PDO
 * @throws PDOException
 */
function getConnection() {
    static $conn = null;
    
    if ($conn === null) {
        try {
            $conn = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . CHARSET,
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false,
                    PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . CHARSET
                ]
            );
        } catch(PDOException $e) {
            if (ENVIRONMENT === 'development') {
                die('Erro na conexão: ' . $e->getMessage());
            } else {
                http_response_code(500);
                die(json_encode([
                    'success' => false,
                    'message' => 'Erro na conexão com o banco de dados'
                ]));
            }
        }
    }
    
    return $conn;
}

/**
 * Retorna uma conexão MySQLi com o banco de dados
 * (Usado por alguns scripts legados e Kwai)
 * 
 * @return mysqli
 */
function getMysqliConnection() {
    static $conn = null;
    
    if ($conn === null) {
        $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
        
        if ($conn->connect_error) {
            if (ENVIRONMENT === 'development') {
                die('Erro na conexão MySQLi: ' . $conn->connect_error);
            } else {
                http_response_code(500);
                die(json_encode([
                    'success' => false,
                    'message' => 'Erro de conexão com banco de dados'
                ]));
            }
        }
        
        $conn->set_charset(CHARSET);
    }
    
    return $conn;
}

// ============================================================================
// FUNÇÃO PARA INICIAR SESSÃO DE FORMA SEGURA
// ============================================================================

/**
 * Inicia a sessão de forma segura e consistente
 */
function initSession() {
    // Verificar se a sessão não está ativa
    if (session_status() !== PHP_SESSION_ACTIVE) {
        // Auto-detectar se está usando HTTPS
        $isSecure = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') || 
                    (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https');
        
        // Configurar diretório de sessões (usar tmp do sistema ou criar um local)
        $sessionPath = session_save_path();
        
        // Se o path padrão não existe, tentar alternativas
        if (empty($sessionPath) || !is_dir($sessionPath) || !is_writable($sessionPath)) {
            // Tentar usar diretório local
            $localSessionPath = DIR_ROOT . '/sessions';
            
            if (!is_dir($localSessionPath)) {
                @mkdir($localSessionPath, 0700, true);
            }
            
            if (is_dir($localSessionPath) && is_writable($localSessionPath)) {
                session_save_path($localSessionPath);
            } else {
                // Usar /tmp como último recurso
                session_save_path('/tmp');
            }
        }
        
        // Configurar nome da sessão ANTES de iniciar
        session_name(SESSION_NAME);
        
        // Configurar parâmetros do cookie
        session_set_cookie_params([
            'lifetime' => SESSION_LIFETIME,
            'path' => SESSION_COOKIE_PATH,
            'domain' => SESSION_COOKIE_DOMAIN,
            'secure' => $isSecure,
            'httponly' => SESSION_COOKIE_HTTPONLY,
            'samesite' => 'Lax'
        ]);
        
        // Configurar garbage collection
        ini_set('session.gc_maxlifetime', SESSION_LIFETIME);
        
        // Iniciar a sessão
        @session_start();
    }
}

// ============================================================================
// HEADERS PADRÃO PARA APIs
// ============================================================================

/**
 * Define headers padrão para respostas de API
 */
function setApiHeaders() {
    header('Content-Type: application/json; charset=utf-8');
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization');
    
    // Handle preflight requests
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(200);
        exit;
    }
}

// ============================================================================
// FUNÇÕES AUXILIARES PARA URLs
// ============================================================================

/**
 * Retorna a URL completa para um caminho relativo
 * 
 * @param string $path Caminho relativo (ex: 'principal/index.html')
 * @return string URL completa
 */
function getUrl($path = '') {
    $path = ltrim($path, '/');
    return URL_COMPLETA . '/' . $path;
}

/**
 * Retorna a URL para um arquivo de upload
 * 
 * @param string $filename Nome do arquivo
 * @return string URL completa do upload
 */
function getUploadUrl($filename) {
    if (empty($filename)) {
        return getUrl('assets/logo.png');
    }
    
    // Se já é uma URL completa (começa com http:// ou https://), retorna como está
    if (preg_match('/^https?:\/\//', $filename)) {
        return $filename;
    }
    
    // Se começa com /, é um caminho absoluto - adicionar apenas URL_BASE
    if (strpos($filename, '/') === 0) {
        return URL_BASE . $filename;
    }
    
    // Se já tem 'uploads/', adiciona BASE_PATH
    if (strpos($filename, 'uploads/') === 0) {
        return URL_BASE . BASE_PATH . $filename;
    }
    
    // Caso contrário, é apenas o nome do arquivo - adiciona caminho completo
    return URL_BASE . URL_UPLOADS . ltrim($filename, '/');
}

/**
 * Retorna a URL para um asset
 * 
 * @param string $filename Nome do arquivo
 * @return string URL completa do asset
 */
function getAssetUrl($filename) {
    $filename = ltrim($filename, '/');
    return URL_BASE . URL_ASSETS . $filename;
}

/**
 * Retorna a URL para um endpoint do backend
 * 
 * @param string $endpoint Nome do arquivo PHP
 * @return string URL completa do endpoint
 */
function getBackendUrl($endpoint) {
    $endpoint = ltrim($endpoint, '/');
    return URL_BASE . URL_BACKEND . $endpoint;
}

// ============================================================================
// FUNÇÃO PARA VERIFICAR AUTENTICAÇÃO
// ============================================================================

/**
 * Verifica se o usuário está autenticado
 * 
 * @param bool $returnJson Se true, retorna JSON em caso de erro
 * @return bool|void
 */
function checkAuth($returnJson = true) {
    initSession();
    
    if (!isset($_SESSION['user_id'])) {
        if ($returnJson) {
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Não autenticado',
                'redirect' => getUrl('autenticacao/index.html')
            ]);
            exit;
        }
        return false;
    }
    
    return true;
}

/**
 * Retorna o ID do usuário logado
 * 
 * @return int|null
 */
function getUserId() {
    initSession();
    return $_SESSION['user_id'] ?? null;
}

// ============================================================================
// CONFIGURAÇÕES KWAI ADS
// ============================================================================

define('KWAI_PIXEL_ID', '294280481040524');
define('KWAI_TEST_TOKEN', '6GkdYNX1HW_KUc5nePoCkQ');
define('KWAI_WEB_EVENT_TOKEN', '0OwMvfbjwazyOd_Op1jISQ5WPIfItCo2ASgVmui4I34');
define('KWAI_API_POST_URL', 'https://www.adsnebula.com/log/common/api');
define('KWAI_API_GET_URL', 'https://www.adsnebula.com/log/common/gapi');
define('KWAI_MMP_CODE', 'PL');
define('KWAI_SDK_VERSION', '9.9.9');
define('KWAI_IS_ATTRIBUTED', 1);
define('KWAI_TEST_MODE', false);
define('KWAI_TRACK_MODE', false);
define('KWAI_DEFAULT_CURRENCY', 'BRL');

// ============================================================================
// FIM DA CONFIGURAÇÃO
// ============================================================================
// 
// IMPORTANTE: Cada arquivo backend deve chamar initSession() explicitamente
// para garantir que a sessão seja iniciada com os parâmetros corretos.
