<?php
/**
 * Verifica token de "lembrar login" e faz login automático
 */

require_once 'config_global.php';

header('Content-Type: application/json');

// Verificar se tem cookie de token
// Só faz auto-login se tiver o cookie "remember_token"
if (!isset($_COOKIE['remember_token'])) {
    echo json_encode([
        'success' => true,
        'logged_in' => false
    ]);
    exit;
}

// Iniciar sessão apenas se tiver o token
session_start();

$token = $_COOKIE['remember_token'];

try {
    $conn = getConnection();
    
    // Buscar token válido
    $stmt = $conn->prepare("
        SELECT t.user_id, u.nome, u.whatsapp, u.perfil_completo, u.bloqueado
        FROM login_tokens t
        INNER JOIN usuarios u ON t.user_id = u.id
        WHERE t.token = ? AND t.expires_at > NOW()
    ");
    
    $stmt->execute([$token]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($result) {
        // Verificar se a conta está bloqueada
        if (isset($result['bloqueado']) && $result['bloqueado'] == 1) {
            // Deletar token e cookie
            setcookie('remember_token', '', time() - 3600, '/');
            $stmt = $conn->prepare("DELETE FROM login_tokens WHERE token = ?");
            $stmt->execute([$token]);
            
            echo json_encode([
                'success' => false,
                'message' => 'Conta bloqueada',
                'blocked' => true
            ]);
            exit;
        }
        
        // Token válido, fazer login automático
        $_SESSION['user_id'] = $result['user_id'];
        $_SESSION['user_nome'] = $result['nome'];
        $_SESSION['user_whatsapp'] = $result['whatsapp'];
        
        // Renovar token por mais 24 horas
        $newExpires = time() + (24 * 60 * 60);
        $stmt = $conn->prepare("
            UPDATE login_tokens 
            SET expires_at = FROM_UNIXTIME(?)
            WHERE token = ?
        ");
        $stmt->execute([$newExpires, $token]);
        
        // Renovar cookie
        setcookie('remember_token', $token, $newExpires, '/', '', false, true);
        
        echo json_encode([
            'success' => true,
            'logged_in' => true,
            'user_id' => $result['user_id'],
            'perfil_completo' => (bool)$result['perfil_completo']
        ]);
    } else {
        // Token inválido ou expirado, remover cookie
        setcookie('remember_token', '', time() - 3600, '/');
        
        echo json_encode([
            'success' => true,
            'logged_in' => false
        ]);
    }
    
} catch(PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro: ' . $e->getMessage()
    ]);
}
?>
