<?php
/**
 * API para admin atualizar foto de perfil de usuário
 */

session_start();

header('Content-Type: application/json');

// Verificar se é admin
if (!isset($_SESSION['admin_authenticated']) || $_SESSION['admin_authenticated'] !== true) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Acesso não autorizado'
    ]);
    exit;
}

require_once 'config.php';

try {
    // Validar dados
    if (!isset($_POST['user_id']) || !isset($_FILES['foto_perfil'])) {
        throw new Exception('Dados incompletos');
    }
    
    $userId = intval($_POST['user_id']);
    $file = $_FILES['foto_perfil'];
    
    // Validar upload
    if ($file['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('Erro no upload do arquivo');
    }
    
    // Validar tipo de arquivo
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mimeType = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);
    
    if (!in_array($mimeType, $allowedTypes)) {
        throw new Exception('Tipo de arquivo não permitido. Use JPG, PNG, GIF ou WEBP');
    }
    
    // Validar tamanho (max 5MB)
    if ($file['size'] > 5 * 1024 * 1024) {
        throw new Exception('Arquivo muito grande. Máximo 5MB');
    }
    
    $conn = getConnection();
    
    // Buscar foto antiga para deletar
    $stmt = $conn->prepare("SELECT foto_perfil FROM usuarios WHERE id = ?");
    $stmt->execute([$userId]);
    $oldPhoto = $stmt->fetchColumn();
    
    // Gerar nome único para o arquivo
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = 'profile_' . $userId . '_' . time() . '.' . $extension;
    $uploadPath = __DIR__ . '/../uploads/' . $filename;
    
    // Mover arquivo
    if (!move_uploaded_file($file['tmp_name'], $uploadPath)) {
        throw new Exception('Erro ao salvar arquivo');
    }
    
    // Atualizar banco de dados
    $relativePath = 'uploads/' . $filename;
    $stmt = $conn->prepare("UPDATE usuarios SET foto_perfil = ? WHERE id = ?");
    $stmt->execute([$relativePath, $userId]);
    
    // Deletar foto antiga se existir
    if ($oldPhoto && file_exists(__DIR__ . '/../' . $oldPhoto)) {
        unlink(__DIR__ . '/../' . $oldPhoto);
    }
    
    // Log da ação
    error_log("Admin {$_SESSION['admin_user_name']} atualizou foto do usuário ID {$userId}");
    
    echo json_encode([
        'success' => true,
        'message' => 'Foto atualizada com sucesso',
        'new_photo' => $relativePath
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
