<?php
/**
 * Admin - Limpar todas as mensagens e mídias
 * Remove todas as mensagens, mídias enviadas e conversas
 */

session_start();

header('Content-Type: application/json');

// Verificar se é admin
if (!isset($_SESSION['admin_authenticated']) || $_SESSION['admin_authenticated'] !== true) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Acesso não autorizado'
    ]);
    exit;
}

require_once 'config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
    exit;
}

try {
    $conn = getConnection();
    
    // Contar antes de deletar (para estatísticas)
    $stmt = $conn->query("SELECT COUNT(*) as total FROM mensagens");
    $totalMensagens = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    $stmt = $conn->query("SELECT COUNT(*) as total FROM conversas");
    $totalConversas = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Buscar todos os caminhos de mídias antes de deletar
    $stmt = $conn->query("SELECT caminho_midia FROM mensagens WHERE caminho_midia IS NOT NULL");
    $midias = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $midiasDeleted = 0;
    $uploadDir = __DIR__ . '/../uploads/';
    
    // Deletar arquivos de mídia
    foreach ($midias as $caminho) {
        $filePath = __DIR__ . '/../' . $caminho;
        if (file_exists($filePath)) {
            if (unlink($filePath)) {
                $midiasDeleted++;
            }
        }
    }
    
    // Deletar registros do banco de dados
    $conn->beginTransaction();
    
    // Deletar todas as mensagens
    $conn->exec("DELETE FROM mensagens");
    
    // Deletar todas as conversas
    $conn->exec("DELETE FROM conversas");
    
    // Resetar os auto_increment para economizar espaço
    $conn->exec("ALTER TABLE mensagens AUTO_INCREMENT = 1");
    $conn->exec("ALTER TABLE conversas AUTO_INCREMENT = 1");
    
    $conn->commit();
    
    // Log da ação
    $adminId = $_SESSION['admin_user_id'] ?? 'unknown';
    $adminName = $_SESSION['admin_user_name'] ?? 'Admin';
    error_log("Admin $adminName (ID: $adminId) limpou todas as mensagens. Total: $totalMensagens mensagens, $totalConversas conversas, $midiasDeleted arquivos deletados.");
    
    echo json_encode([
        'success' => true,
        'message' => "Limpeza concluída! $totalMensagens mensagens, $totalConversas conversas e $midiasDeleted arquivos deletados.",
        'stats' => [
            'mensagens' => $totalMensagens,
            'conversas' => $totalConversas,
            'midias' => $midiasDeleted
        ]
    ]);
    
} catch (Exception $e) {
    if (isset($conn) && $conn->inTransaction()) {
        $conn->rollBack();
    }
    
    error_log("Erro ao limpar mensagens: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao limpar mensagens: ' . $e->getMessage()
    ]);
}
?>
