// Toggle between login and register forms
function toggleForms() {
    const loginForm = document.getElementById('loginForm');
    const registerForm = document.getElementById('registerForm');
    
    loginForm.classList.toggle('hidden');
    registerForm.classList.toggle('hidden');
}

// Toggle password visibility
function togglePassword(inputId) {
    const input = document.getElementById(inputId);
    const button = input.parentElement.querySelector('.toggle-password');
    const icon = button.querySelector('i');
    
    if (input.type === 'password') {
        input.type = 'text';
        icon.classList.remove('fa-eye');
        icon.classList.add('fa-eye-slash');
    } else {
        input.type = 'password';
        icon.classList.remove('fa-eye-slash');
        icon.classList.add('fa-eye');
    }
}

// Format WhatsApp input
function formatWhatsApp(input) {
    let value = input.value.replace(/\D/g, '');
    
    if (value.length > 11) {
        value = value.slice(0, 11);
    }
    
    if (value.length > 6) {
        value = `(${value.slice(0, 2)}) ${value.slice(2, 7)}-${value.slice(7)}`;
    } else if (value.length > 2) {
        value = `(${value.slice(0, 2)}) ${value.slice(2)}`;
    } else if (value.length > 0) {
        value = `(${value}`;
    }
    
    input.value = value;
}

// Add event listeners for WhatsApp formatting
document.addEventListener('DOMContentLoaded', function() {
    // Verificar auto-login ao carregar a página
    checkAutoLogin();
    
    const whatsappInputs = document.querySelectorAll('input[type="tel"]');
    
    whatsappInputs.forEach(input => {
        input.addEventListener('input', function() {
            formatWhatsApp(this);
        });
    });
    
    // Handle Login Form Submission
    document.getElementById('loginForm').querySelector('form').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const whatsapp = document.getElementById('loginWhatsApp').value;
        const password = document.getElementById('loginPassword').value;
        const rememberMe = document.getElementById('rememberMe').checked;
        const submitBtn = this.querySelector('.btn-submit');
        
        // Validate WhatsApp
        if (whatsapp.replace(/\D/g, '').length < 11) {
            showNotification('Por favor, insira um número de WhatsApp válido', 'error');
            return;
        }
        
        // Disable button and show loading
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<span>Entrando...</span><i class="fas fa-spinner fa-spin"></i>';
        
        try {
            const response = await fetch(CONFIG.api.login, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    whatsapp: whatsapp.replace(/\D/g, ''),
                    senha: password,
                    remember: rememberMe
                })
            });
            
            const data = await response.json();
            
            console.log('Response:', data); // Debug
            
            if (data.success) {
                showNotification(data.message, 'success');
                
                // Verificar se o perfil está completo
                setTimeout(() => {
                    if (data.redirect) {
                        window.location.href = data.redirect;
                    } else if (data.perfil_completo) {
                        redirectTo('principal');
                    } else {
                        redirectTo('completeProfile');
                    }
                }, 1500);
            } else {
                showNotification(data.message, 'error');
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<span>Acessar plataforma</span><i class="fas fa-arrow-right"></i>';
            }
            
        } catch (error) {
            console.error('Erro:', error); // Debug
            showNotification('Erro ao conectar com o servidor', 'error');
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<span>Acessar plataforma</span><i class="fas fa-arrow-right"></i>';
        }
    });
    
    // Handle Register Form Submission
    document.getElementById('registerForm').querySelector('form').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const name = document.getElementById('registerName').value;
        const whatsapp = document.getElementById('registerWhatsApp').value;
        const password = document.getElementById('registerPassword').value;
        const submitBtn = this.querySelector('.btn-submit');
        
        // Validations
        if (whatsapp.replace(/\D/g, '').length < 11) {
            showNotification('Por favor, insira um número de WhatsApp válido', 'error');
            return;
        }
        
        if (password.length < 8) {
            showNotification('A senha deve ter no mínimo 8 caracteres', 'error');
            return;
        }
        
        // Disable button and show loading
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<span>Criando conta...</span><i class="fas fa-spinner fa-spin"></i>';
        
        try {
            // Use caminho absoluto a partir da raiz do domínio
            const response = await fetch('/mensagem/backend/register.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    nome: name,
                    whatsapp: whatsapp.replace(/\D/g, ''),
                    senha: password
                })
            });
            
            const data = await response.json();
            
            console.log('Response:', data); // Debug
            
            if (data.success) {
                showNotification(data.message, 'success');
                
                // Kwai Tracking: Evento de registro completo
                if (typeof kwaiTracker !== 'undefined') {
                    kwaiTracker.trackRegistration();
                }
                
                // Redirect to complete profile after 1.5s
                setTimeout(() => {
                    if (data.redirect) {
                        window.location.href = data.redirect;
                    } else {
                        window.location.href = '../complete_profile/index.html';
                    }
                }, 1500);
            } else {
                showNotification(data.message, 'error');
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<span>Criar minha conta</span><i class="fas fa-arrow-right"></i>';
            }
            
        } catch (error) {
            console.error('Erro:', error); // Debug
            showNotification('Erro ao conectar com o servidor', 'error');
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<span>Criar minha conta</span><i class="fas fa-arrow-right"></i>';
        }
    });
});

// Notification system
function showNotification(message, type = 'info') {
    // Remove existing notification
    const existing = document.querySelector('.notification');
    if (existing) {
        existing.remove();
    }
    
    // Create notification
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
        <span>${message}</span>
    `;
    
    // Add styles
    notification.style.cssText = `
        position: fixed;
        top: 24px;
        right: 24px;
        background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
        color: white;
        padding: 16px 24px;
        border-radius: 12px;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        display: flex;
        align-items: center;
        gap: 12px;
        font-size: 14px;
        font-weight: 600;
        z-index: 9999;
        animation: slideInRight 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 3s
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

// Add animation styles
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from {
            transform: translateX(400px);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    @keyframes slideOutRight {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(400px);
            opacity: 0;
        }
    }
`;
document.head.appendChild(style);

// ===== AUTO-LOGIN =====
async function checkAutoLogin() {
    try {
        const response = await fetch('/mensagem/backend/check_auto_login.php');
        const data = await response.json();
        
        if (data.success && data.logged_in) {
            // Usuário já está logado, redirecionar
            if (data.perfil_completo) {
                window.location.href = '/mensagem/principal/index.html';
            } else {
                window.location.href = '/mensagem/complete_profile/index.html';
            }
        }
    } catch (error) {
        console.error('Erro ao verificar auto-login:', error);
    }
}

// ===== MODAL DE RECUPERAÇÃO DE SENHA =====
function openForgotPasswordModal() {
    const modal = document.getElementById('forgotPasswordModal');
    modal.style.display = 'flex';
    document.body.style.overflow = 'hidden';
}

function closeForgotPasswordModal() {
    const modal = document.getElementById('forgotPasswordModal');
    modal.style.display = 'none';
    document.body.style.overflow = 'auto';
    
    // Limpar campos
    document.getElementById('resetWhatsApp').value = '';
    document.getElementById('newPassword').value = '';
    document.getElementById('confirmPassword').value = '';
}

// Fechar modal ao clicar fora
document.addEventListener('DOMContentLoaded', function() {
    const modal = document.getElementById('forgotPasswordModal');
    if (modal) {
        modal.addEventListener('click', function(e) {
            if (e.target === modal) {
                closeForgotPasswordModal();
            }
        });
    }
});

// Handle Forgot Password Form
document.addEventListener('DOMContentLoaded', function() {
    const forgotForm = document.getElementById('forgotPasswordForm');
    if (forgotForm) {
        forgotForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const whatsapp = document.getElementById('resetWhatsApp').value;
            const newPassword = document.getElementById('newPassword').value;
            const confirmPassword = document.getElementById('confirmPassword').value;
            const submitBtn = this.querySelector('.btn-submit');
            
            // Validações
            if (whatsapp.replace(/\D/g, '').length < 11) {
                showNotification('Por favor, insira um número de WhatsApp válido', 'error');
                return;
            }
            
            if (newPassword.length < 8) {
                showNotification('A senha deve ter no mínimo 8 caracteres', 'error');
                return;
            }
            
            if (newPassword !== confirmPassword) {
                showNotification('As senhas não coincidem', 'error');
                return;
            }
            
            // Loading
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<span>Alterando senha...</span><i class="fas fa-spinner fa-spin"></i>';
            
            try {
                const response = await fetch('/mensagem/backend/reset_password.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        whatsapp: whatsapp.replace(/\D/g, ''),
                        nova_senha: newPassword
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showNotification(data.message, 'success');
                    closeForgotPasswordModal();
                    
                    // Limpar campos de login
                    document.getElementById('loginWhatsApp').value = '';
                    document.getElementById('loginPassword').value = '';
                } else {
                    showNotification(data.message, 'error');
                }
                
            } catch (error) {
                console.error('Error:', error);
                showNotification('Erro ao alterar senha. Tente novamente.', 'error');
            } finally {
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<span>Alterar Senha</span><i class="fas fa-check"></i>';
            }
        });
    }
});
