# Sistema de Usuários Online

## 🎯 Funcionalidades Implementadas

### Backend

1. **`update_online_status.php`**
   - Atualiza o campo `last_seen` do usuário com timestamp atual
   - Chamado automaticamente a cada 30 segundos
   - Usado para determinar quem está online

2. **`get_online_users.php`**
   - Retorna lista de usuários online (vistos nos últimos 2 minutos)
   - Suporta filtro por gênero: todos, masculino, feminino, outro
   - Exclui o próprio usuário da lista
   - Retorna: id, nome, idade, sexo, foto_perfil

3. **`update_table_online.php`**
   - Script de migração para adicionar campo `last_seen`
   - **Execute UMA VEZ** acessando pelo navegador

### Frontend

1. **Interface de Usuários Online**
   - Cards com foto, nome, idade e sexo
   - Indicador verde pulsante mostrando status online
   - Botão para iniciar conversa
   - Filtros por gênero (Todos/Homens/Mulheres/Outros)

2. **Atualizações Automáticas**
   - Status online atualizado a cada 30 segundos
   - Lista de usuários atualizada a cada 15 segundos
   - Limpa intervals ao sair da página

3. **Sistema de Presença**
   - Usuário é considerado online se foi visto nos últimos 2 minutos
   - Atualização silenciosa em background
   - Não requer ação do usuário

## 🚀 Como Usar

### 1. Executar Migração do Banco

Acesse no navegador:
```
https://prattix.com/mensagem/backend/update_table_online.php
```

Isso adiciona o campo `last_seen` na tabela `usuarios`.

### 2. Funcionamento Automático

Quando o usuário entra na página `/principal`:
- ✅ Marca-se como online automaticamente
- ✅ Vê outros usuários online
- ✅ Pode filtrar por gênero
- ✅ Pode clicar no botão de chat (a ser implementado)

## 📊 Como Funciona

### Fluxo de Presença

1. Usuário acessa `/principal/index.html`
2. JavaScript executa `updateOnlineStatus()` imediatamente
3. Backend atualiza `last_seen = NOW()`
4. A cada 30 segundos, repete o processo
5. Usuário é considerado offline após 2 minutos sem atualizar

### Fluxo de Listagem

1. JavaScript executa `loadOnlineUsers()` com filtro
2. Backend busca usuários onde:
   - `last_seen >= NOW() - 2 minutos`
   - `perfil_completo = 1`
   - `id != usuário_atual`
   - Filtro de gênero (se aplicado)
3. Retorna array de usuários
4. Frontend renderiza cards
5. A cada 15 segundos, atualiza a lista

## 🎨 Design dos Cards

### Estrutura HTML (gerado dinamicamente)
```html
<div class="user-card" data-user-id="123">
    <div class="user-avatar">
        <img src="/mensagem/uploads/foto.jpg" alt="Nome">
        <div class="online-dot"></div>
    </div>
    <div class="user-info">
        <div class="user-name">Nome do Usuário</div>
        <div class="user-details">
            <span><i class="fas fa-calendar"></i> 25 anos</span>
            <span><i class="fas fa-mars"></i> Masculino</span>
        </div>
    </div>
    <button class="chat-btn" onclick="iniciarChat(123, 'Nome')">
        <i class="fas fa-comment"></i>
    </button>
</div>
```

### Estilos CSS
- Foto de perfil redonda com borda verde
- Indicador online verde pulsante
- Hover com elevação e borda
- Botão de chat gradiente verde
- Animação suave em todas as interações

## 🔄 Parâmetros da API

### GET `/backend/get_online_users.php`

**Query Parameters:**
- `filtro` (opcional): 
  - `todos` (padrão)
  - `masculino`
  - `feminino`
  - `outro`

**Exemplo:**
```
/mensagem/backend/get_online_users.php?filtro=feminino
```

**Resposta:**
```json
{
  "success": true,
  "usuarios": [
    {
      "id": 5,
      "nome": "Maria Silva",
      "idade": 28,
      "sexo": "feminino",
      "foto_perfil": "uploads/12345_foto.jpg",
      "online": true
    }
  ],
  "total": 1
}
```

### POST `/backend/update_online_status.php`

**Body:** Nenhum (usa sessão)

**Resposta:**
```json
{
  "success": true,
  "timestamp": 1699024800
}
```

## 📝 Próximos Passos

### Sugestões de Implementação

1. **Sistema de Chat**
   - Criar tabela `conversas` e `mensagens`
   - WebSocket para chat em tempo real
   - Notificações de novas mensagens

2. **Melhorias de Performance**
   - Paginação da lista de usuários
   - Cache de resultados por alguns segundos
   - Lazy loading de fotos

3. **Recursos Adicionais**
   - Última vez visto (ex: "Online há 5 minutos")
   - Status personalizado
   - Bloquear/reportar usuários
   - Busca por nome

4. **Otimizações**
   - Usar WebSocket em vez de polling
   - Implementar Server-Sent Events (SSE)
   - Adicionar Service Worker para funcionar offline

## 🐛 Solução de Problemas

### Usuários não aparecem online

1. Verificar se executou `update_table_online.php`
2. Verificar se `perfil_completo = 1` no banco
3. Checar console do navegador por erros
4. Verificar se usuário está de fato acessando a página

### Lista não atualiza

1. Verificar se intervals estão rodando (console.log)
2. Limpar cache do navegador (Ctrl + F5)
3. Verificar sessão PHP ativa
4. Checar logs de erro do servidor

### Performance lenta

1. Reduzir frequência de atualização (aumentar intervalos)
2. Limitar número de usuários retornados
3. Adicionar índices no banco: `CREATE INDEX idx_last_seen ON usuarios(last_seen)`

## 💡 Dicas

- O sistema funciona completamente em background
- Não precisa de ação do usuário
- Compatível com mobile e desktop
- Leve e performático (requisições pequenas)
