<?php
require_once 'config.php';

header('Content-Type: application/json; charset=utf-8');

try {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($data['name']) || !isset($data['whatsapp'])) {
        throw new Exception('Nome e WhatsApp são obrigatórios');
    }
    
    $name = trim($data['name']);
    $whatsapp = preg_replace('/[^0-9]/', '', $data['whatsapp']); // Remove tudo exceto números
    
    // Validações
    if (empty($name) || strlen($name) < 3) {
        throw new Exception('Nome deve ter pelo menos 3 caracteres');
    }
    
    if (empty($whatsapp) || strlen($whatsapp) < 10) {
        throw new Exception('WhatsApp inválido');
    }
    
    $db = Database::getInstance()->getConnection();
    
    // Verificar se usuário já existe
    $stmt = $db->prepare("SELECT id, name, balance FROM users WHERE whatsapp = ?");
    $stmt->execute([$whatsapp]);
    $existingUser = $stmt->fetch();
    
    if ($existingUser) {
        // Usuário já existe, verificar similaridade do nome para login
        $dbName = strtolower(trim($existingUser['name']));
        $inputName = strtolower(trim($name));
        
        // Verificar se tem pelo menos 3 letras sequenciais em comum
        $hasMatch = false;
        $minLength = 3;
        
        for ($i = 0; $i <= strlen($inputName) - $minLength; $i++) {
            $substring = substr($inputName, $i, $minLength);
            if (strpos($dbName, $substring) !== false) {
                $hasMatch = true;
                break;
            }
        }
        
        if (!$hasMatch) {
            throw new Exception('Nome não corresponde ao cadastrado neste WhatsApp');
        }
        
        // Usuário já existe, fazer login
        $userId = $existingUser['id'];
        
        // Atualizar último login
        $stmt = $db->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
        $stmt->execute([$userId]);
        
    } else {
        // Criar novo usuário
        $stmt = $db->prepare("INSERT INTO users (name, whatsapp, balance, created_at) VALUES (?, ?, 0.00, NOW())");
        $stmt->execute([$name, $whatsapp]);
        $userId = $db->lastInsertId();
        
        // Log de auditoria
        $stmt = $db->prepare("INSERT INTO audit_logs (user_id, action, description, ip_address) VALUES (?, 'register', 'Novo usuário cadastrado', ?)");
        $stmt->execute([$userId, $_SERVER['REMOTE_ADDR'] ?? 'unknown']);
    }
    
    // Criar sessão
    $sessionToken = bin2hex(random_bytes(32));
    $expiresAt = date('Y-m-d H:i:s', time() + SESSION_DURATION);
    
    $stmt = $db->prepare("INSERT INTO sessions (user_id, session_token, ip_address, user_agent, expires_at) VALUES (?, ?, ?, ?, ?)");
    $stmt->execute([
        $userId,
        $sessionToken,
        $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
        $expiresAt
    ]);
    
    // Buscar dados atualizados do usuário
    $stmt = $db->prepare("SELECT id, name, whatsapp, balance, total_deposited FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    echo json_encode([
        'success' => true,
        'message' => $existingUser ? 'Login realizado com sucesso' : 'Cadastro realizado com sucesso',
        'user' => [
            'id' => $user['id'],
            'name' => $user['name'],
            'whatsapp' => $user['whatsapp'],
            'balance' => floatval($user['balance']),
            'total_deposited' => floatval($user['total_deposited'])
        ],
        'session_token' => $sessionToken
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
