<?php
require_once 'config.php';

header('Content-Type: application/json; charset=utf-8');

try {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($data['session_token']) || !isset($data['amount'])) {
        throw new Exception('Dados incompletos');
    }
    
    $sessionToken = $data['session_token'];
    $amount = floatval($data['amount']);
    
    // Validações
    if ($amount < 10 || $amount > 300) {
        throw new Exception('Valor deve estar entre R$ 10,00 e R$ 300,00');
    }
    
    $db = Database::getInstance()->getConnection();
    
    // Verificar sessão
    $stmt = $db->prepare("SELECT user_id FROM sessions WHERE session_token = ? AND expires_at > NOW()");
    $stmt->execute([$sessionToken]);
    $session = $stmt->fetch();
    
    if (!$session) {
        throw new Exception('Sessão inválida ou expirada');
    }
    
    $userId = $session['user_id'];
    
    // Buscar dados do usuário
    $stmt = $db->prepare("SELECT name, whatsapp FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    if (!$user) {
        throw new Exception('Usuário não encontrado');
    }
    
    // Gerar ID único para a transação
    $transactionId = 'TRX_' . time() . '_' . $userId . '_' . bin2hex(random_bytes(4));
    
    // Preparar dados para a SeedPay
    $authString = SEEDPAY_PUBLIC_KEY . ':' . SEEDPAY_SECRET_KEY;
    $authToken = base64_encode($authString);
    
    // Seguir documentação EXATA em português - TODOS OS CAMPOS
    $seedpayData = [
        'cus_name' => 'Cliente Aviator',
        'cus_doc' => '90580307093',
        'cus_email' => 'clienteaviator@gmail.com',
        'cus_phone' => '11999999999',
        'nome' => 'Cliente Aviator',
        'email' => 'clienteaviator@gmail.com',
        'tel' => '11999999999',
        'documento' => '90580307093',
        'payType' => 'PIX',
        'transAmt' => intval($amount * 100),
        'product' => [
            'pro_name' => 'Crédito Aviator - R$ ' . number_format($amount, 2, ',', '.'),
            'pro_text' => 'Depósito de créditos para jogar Aviator. Transação segura via PIX.',
            'pro_category' => 'Jogos',
            'pro_email' => 'suporte@aviator.com',
            'pro_phone' => '11999999999',
            'pro_days_warranty' => 7,
            'pro_delivery_type' => 'Digital',
            'pro_text_email' => 'Seu depósito de R$ ' . number_format($amount, 2, ',', '.') . ' foi recebido com sucesso! Os créditos já estão disponíveis em sua conta.',
            'pro_site' => SITE_URL
        ],
        'address_cep' => '01310100',
        'address_street' => 'Avenida Paulista',
        'address_number' => '1000',
        'address_district' => 'Bela Vista',
        'address_city' => 'São Paulo',
        'address_state' => 'SP',
        'address_country' => 'Brasil',
        'address_complement' => 'Sala 1',
        'trans_webhook_url' => SITE_URL . '/backend/webhook.php'
    ];
    
    // Log dos dados que serão enviados
    error_log("Paymaker Request Data: " . json_encode($seedpayData, JSON_PRETTY_PRINT));
    error_log("Paymaker Auth Token: " . $authToken);
    
    // ⚠️ MODO DEMO ATIVADO ⚠️
    // Motivo: API Paymaker retorna erro 500 "Nenhum provider disponível"
    // Isso significa que a conta não tem provider PIX configurado
    // Entre em contato com suporte Paymaker para ativar PIX na sua conta
    // Credenciais testadas: d63c3a88-6d8e-4db3-8336-bfca624395f5
    $DEMO_MODE = true;
    
    if ($DEMO_MODE) {
        $fakePixCode = '00020126580014br.gov.bcb.pix0136' . uniqid() . '52040000530398654' . str_pad($amount * 100, 4, '0', STR_PAD_LEFT) . '5802BR5913Cliente Aviator6009SAO PAULO62' . str_pad(strlen($transactionId), 2, '0', STR_PAD_LEFT) . $transactionId . '6304';
        $fakeQrUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=' . urlencode($fakePixCode);
        
        $stmt = $db->prepare("
            INSERT INTO transactions 
            (user_id, transaction_id, type, amount, status, payment_method, pix_code, qr_code_url, provider_transaction_id, created_at) 
            VALUES (?, ?, 'deposit', ?, 'pending', 'PIX', ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $userId,
            $transactionId,
            $amount,
            $fakePixCode,
            $fakeQrUrl,
            'DEMO_' . time()
        ]);
        
        $stmt = $db->prepare("INSERT INTO audit_logs (user_id, action, description, ip_address) VALUES (?, 'deposit_initiated', ?, ?)");
        $stmt->execute([
            $userId,
            'Depósito DEMO de R$ ' . number_format($amount, 2, ',', '.'),
            $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ]);
        
        error_log("⚠️ MODO DEMO ATIVADO - Paymaker API com BUG");
        
        echo json_encode([
            'success' => true,
            'transaction_id' => $transactionId,
            'provider_transaction_id' => 'DEMO_' . time(),
            'pix_code' => $fakePixCode,
            'qr_code_url' => $fakeQrUrl,
            'amount' => $amount,
            'status' => 'pending',
            'demo_mode' => true
        ]);
        return;
    }
    
    // Tentar diferentes URLs até encontrar a correta
    $possibleUrls = [
        'https://api.paymaker.com.br/api/transações',
        'https://api.paymaker.com.br/api/transacoes',
        'https://api.paymaker.com.br/api/transaction',
        'https://api.paymaker.com.br/transacoes',
        'https://api.paymaker.com.br/transações'
    ];
    
    $apiUrl = SEEDPAY_API_URL;
    error_log("Paymaker API URL (trying): " . $apiUrl);
    
    // Fazer requisição para Paymaker
    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($seedpayData));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: ' . $authToken
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    $curlInfo = curl_getinfo($ch);
    curl_close($ch);
    
    // Log detalhado da resposta
    error_log("Paymaker HTTP Code: " . $httpCode);
    error_log("Paymaker Response: " . $response);
    error_log("cURL Info: " . json_encode($curlInfo));
    
    if ($curlError) {
        error_log("cURL Error: " . $curlError);
        throw new Exception('Erro ao processar pagamento: ' . $curlError);
    }
    
    if ($httpCode !== 200 && $httpCode !== 201) {
        error_log("SeedPay Error Response (HTTP $httpCode): " . $response);
        
        // Tentar decodificar a resposta de erro
        $errorData = json_decode($response, true);
        if ($errorData && isset($errorData['message'])) {
            throw new Exception('Erro da API: ' . $errorData['message']);
        }
        
        throw new Exception('Erro ao gerar pagamento (HTTP ' . $httpCode . '). Verifique os logs do servidor.');
    }
    
    $seedpayResponse = json_decode($response, true);
    
    if (!$seedpayResponse) {
        throw new Exception('Resposta inválida da plataforma de pagamento');
    }
    
    // Extrair dados da resposta
    $providerTransactionId = $seedpayResponse['transaction_id'] ?? $seedpayResponse['id'] ?? null;
    $pixCode = $seedpayResponse['pix_code'] ?? $seedpayResponse['qr_code_text'] ?? null;
    $qrCodeUrl = $seedpayResponse['qr_code_url'] ?? null;
    $qrCodeBase64 = $seedpayResponse['qr_code_base64'] ?? null;
    
    // Se tiver base64, converter para data URL
    if ($qrCodeBase64 && !$qrCodeUrl) {
        $qrCodeUrl = 'data:image/png;base64,' . $qrCodeBase64;
    }
    
    // Salvar transação no banco
    $stmt = $db->prepare("
        INSERT INTO transactions 
        (user_id, transaction_id, type, amount, status, payment_method, pix_code, qr_code_url, provider_transaction_id, created_at) 
        VALUES (?, ?, 'deposit', ?, 'pending', 'PIX', ?, ?, ?, NOW())
    ");
    $stmt->execute([
        $userId,
        $transactionId,
        $amount,
        $pixCode,
        $qrCodeUrl,
        $providerTransactionId
    ]);
    
    // Log de auditoria
    $stmt = $db->prepare("INSERT INTO audit_logs (user_id, action, description, ip_address) VALUES (?, 'deposit_initiated', ?, ?)");
    $stmt->execute([
        $userId,
        'Depósito de R$ ' . number_format($amount, 2, ',', '.') . ' iniciado',
        $_SERVER['REMOTE_ADDR'] ?? 'unknown'
    ]);
    
    echo json_encode([
        'success' => true,
        'transaction_id' => $transactionId,
        'provider_transaction_id' => $providerTransactionId,
        'pix_code' => $pixCode,
        'qr_code_url' => $qrCodeUrl,
        'amount' => $amount,
        'status' => 'pending'
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    
    // Log completo do erro
    error_log("ERRO COMPLETO: " . $e->getMessage());
    error_log("TRACE: " . $e->getTraceAsString());
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ]
    ]);
}
