<?php
require_once __DIR__ . '/config.php';

header('Content-Type: application/json; charset=utf-8');

try {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($data['user_id']) || !isset($data['new_balance'])) {
        throw new Exception('Dados incompletos');
    }
    
    $userId = intval($data['user_id']);
    $newBalance = floatval($data['new_balance']);
    
    if ($userId <= 0) {
        throw new Exception('ID de usuário inválido');
    }
    
    if ($newBalance < 0) {
        throw new Exception('Saldo não pode ser negativo');
    }
    
    $db = Database::getInstance()->getConnection();
    
    // Buscar saldo anterior
    $stmt = $db->prepare("SELECT name, balance FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    if (!$user) {
        throw new Exception('Usuário não encontrado');
    }
    
    $oldBalance = floatval($user['balance']);
    
    // Atualizar saldo
    $stmt = $db->prepare("UPDATE users SET balance = ? WHERE id = ?");
    $stmt->execute([$newBalance, $userId]);
    
    // Registrar no histórico de auditoria
    $difference = $newBalance - $oldBalance;
    $action = $difference >= 0 ? 'balance_increase' : 'balance_decrease';
    $description = sprintf(
        'Saldo alterado de R$ %.2f para R$ %.2f (Diferença: %s R$ %.2f) via painel admin',
        $oldBalance,
        $newBalance,
        $difference >= 0 ? '+' : '',
        $difference
    );
    
    $stmt = $db->prepare("INSERT INTO audit_logs (user_id, action, description, ip_address) VALUES (?, ?, ?, ?)");
    $stmt->execute([
        $userId,
        $action,
        $description,
        $_SERVER['REMOTE_ADDR'] ?? 'unknown'
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Saldo atualizado com sucesso',
        'old_balance' => $oldBalance,
        'new_balance' => $newBalance,
        'difference' => $difference
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    error_log("Erro PDO admin_update_balance.php: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => 'Erro no banco de dados: ' . $e->getMessage()
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
