<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/config.php';

try {
    // Pegar dados do POST
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    $response = ['debug' => []];
    $response['debug'][] = 'Input recebido: ' . $input;
    $response['debug'][] = 'Dados decodificados: ' . print_r($data, true);
    
    if (!isset($data['session_token'])) {
        throw new Exception('session_token não enviado');
    }
    
    if (!isset($data['amount'])) {
        throw new Exception('amount não enviado');
    }
    
    $sessionToken = trim($data['session_token']);
    $amount = floatval($data['amount']);
    
    $response['debug'][] = "Session token: $sessionToken";
    $response['debug'][] = "Amount: $amount";
    
    if (empty($sessionToken)) {
        throw new Exception('Token de sessão vazio após trim');
    }
    
    if ($amount <= 0) {
        throw new Exception('Valor inválido: ' . $amount);
    }
    
    $db = Database::getInstance()->getConnection();
    $response['debug'][] = 'Conexão com banco OK';
    
    // Verificar sessão
    $stmt = $db->prepare("SELECT user_id FROM sessions WHERE session_token = ? AND expires_at > NOW()");
    $stmt->execute([$sessionToken]);
    $session = $stmt->fetch();
    
    $response['debug'][] = 'Query sessão executada';
    $response['debug'][] = 'Resultado: ' . print_r($session, true);
    
    if (!$session) {
        // Verificar se a sessão existe mas expirou
        $stmt = $db->prepare("SELECT user_id, expires_at FROM sessions WHERE session_token = ?");
        $stmt->execute([$sessionToken]);
        $expiredSession = $stmt->fetch();
        
        if ($expiredSession) {
            throw new Exception('Sessão expirada em: ' . $expiredSession['expires_at']);
        } else {
            throw new Exception('Sessão não encontrada no banco');
        }
    }
    
    $userId = $session['user_id'];
    $response['debug'][] = "User ID encontrado: $userId";
    
    // Verificar saldo atual
    $stmt = $db->prepare("SELECT balance, total_deposited FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $currentUser = $stmt->fetch();
    $response['debug'][] = "Saldo atual: " . $currentUser['balance'];
    
    // Adicionar saldo
    $stmt = $db->prepare("UPDATE users SET balance = balance + ?, total_deposited = total_deposited + ? WHERE id = ?");
    $result = $stmt->execute([$amount, $amount, $userId]);
    $response['debug'][] = "UPDATE executado: " . ($result ? 'SIM' : 'NÃO');
    $response['debug'][] = "Linhas afetadas: " . $stmt->rowCount();
    
    // Buscar novo saldo
    $stmt = $db->prepare("SELECT balance, total_deposited FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    $response['debug'][] = "Novo saldo: " . $user['balance'];
    
    // Registrar transação
    $transactionId = 'DEP_' . time() . '_' . $userId . '_' . bin2hex(random_bytes(4));
    $providerId = 'CHECKOUT_' . time() . '_' . bin2hex(random_bytes(4));
    
    $response['debug'][] = "Transaction ID: $transactionId";
    $response['debug'][] = "Provider ID: $providerId";
    
    $stmt = $db->prepare("
        INSERT INTO transactions 
        (user_id, transaction_id, type, amount, status, payment_method, provider_transaction_id, created_at) 
        VALUES (?, ?, 'deposit', ?, 'approved', 'Checkout', ?, NOW())
    ");
    $stmt->execute([$userId, $transactionId, $amount, $providerId]);
    $response['debug'][] = "Transação registrada - ID: " . $db->lastInsertId();
    
    // Log de auditoria
    $stmt = $db->prepare("INSERT INTO audit_logs (user_id, action, description, ip_address) VALUES (?, 'deposit_approved', ?, ?)");
    $stmt->execute([
        $userId,
        'Depósito de R$ ' . number_format($amount, 2, ',', '.') . ' aprovado via checkout',
        $_SERVER['REMOTE_ADDR'] ?? 'unknown'
    ]);
    $response['debug'][] = "Auditoria registrada";
    
    $response['success'] = true;
    $response['new_balance'] = floatval($user['balance']);
    $response['amount_added'] = $amount;
    $response['old_balance'] = floatval($currentUser['balance']);
    
    echo json_encode($response, JSON_PRETTY_PRINT);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error_type' => 'PDOException',
        'message' => $e->getMessage(),
        'code' => $e->getCode(),
        'debug' => $response['debug'] ?? []
    ], JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error_type' => 'Exception',
        'message' => $e->getMessage(),
        'debug' => $response['debug'] ?? []
    ], JSON_PRETTY_PRINT);
}
